#pragma rtGlobals=2		// Use modern global access method.
#pragma version=0.10		// 4/8/2004
#include <File Name Utilities>
#include "mafutils", menus=0

//Pattern structure
//	NUMROWS		number of steps in pattern
//	SAMPINT		sampling interval
//	SAVECHAN		list representation of AD channels, e.g. "AD0,AD3"
//	CONTROL		list representation of DA channels, e.g. "DA0,TTL"
//	DEPENDENCY	list of other patterns this one depends on
//	WAVEDEPENDENCY	list of waves this patter depends on
//	PATTYPE		type of pattern: may be "Interval" or "Event"
//	DYNAMICCLAMP	set to 1 for dynamic clamp
//		requires DCVREV1, DCVREV2, DCMINI1, DCMINI2
//	
//
//	Interval patterns have the following fields:
//	TIMEN			duration of step N
//	if TIMEN = "TRAIN"
//		PATTERNN	name of train pattern
//		REPN		number of repetitions for train
//	else
//		AMPN_ch	amplitude for channel ch
//		if AMPN_ch = "RAMP"
//			RAMPSTARTN_ch	start amplitude
//			RAMPENDN_ch		end amplitude
//		if AMPN_ch = "WAVE"
//			WAVEN_ch			name of wave
//		ABSN_ch	Boolean for whether amplitude is absolute or relative to holding potential
//		TTLN		bitwise representation of TTL channels
//		
//	TIMEN, REPN, and AMPN_ch may all contain numbers or parameters
//	If ABSN_ch is 0 for a channel, then the variable HPch (holding potential) must be specified.
//
//	Event patterns have the following fields:
//	TIMEN			time of occurrence of step N
//						if a list, then event occurs at each time in the list
//	EVENTN		type of event: may be "AMP", "TRAIN", "DCPULSE", "TTL", or "FINISH", or "PADFINISH"
//	if EVENTN = "TRAIN"
//		PATTERNN	name of train pattern
//		REPN		number of repetitions for train
//	else
//		CHN		channel for event, i.e. DA channel for amp, bit for TTL, or conductance for DCPULSE
//		AMPN		amplitude (for AMP), 0/1 (for TTL), peak conductance for DCPULSE
//		if AMPN = "WAVE"
//			WAVEN		name of the wave
//		ABSN		determines of AMPN is absolute or relative to holding potential for AMP, RAMPSTART, RAMPEND, and WAVE
//
//	TIMEN, REPN, and AMPN may all contain numbers or parameters
//	In addition, if TIMEN evaluates to a list, then the event is duplicated at each time specified
//		If AMPN evaluates to a list of the same length, then the amplitudes are pulled from that list
//		Note that these lists, if passed in the parameter list must be comma-delimited, and may not contain further variables
//	FINISH provides the absolute time to end
//	PADFINISH ends the specified time after the final event
//
// Improvements to be made:
// *	Properly disable buttons and fields when interval step is a train, and reenable when it isn't

menu "Acquisition"
	"New Set...", /Q, mafPC_NewSet ()
	"Open Set...", /Q, mafPC_OpenSet ("")
	"Save Set", /Q, mafPC_SaveSet (root:maf:mafPC_SetName)
	"Save Set As...", /Q, mafPC_SaveSet ("")
	"Show Set", /Q, mafPC_ShowSet ()
	"-"
end

function mafPC_Init()
	if (!datafolderexists ("root:maf"))
		newdatafolder root:maf
	endif
	if (!exists ("root:maf:mafPC_Patterns"))
		make /t/n=0 root:maf:mafPC_Patterns
		string /g root:maf:mafPC_patternName, root:maf:mafPC_SetName=""
		string /g root:maf:mafPC_param1, root:maf:mafPC_param2
		string /g root:maf:mafPC_mini1, root:maf:mafPC_mini2
		string /g root:maf:mafPC_SaveChan, root:maf:mafPC_Control
		variable /g root:maf:mafPC_sampInt, root:maf:mafPC_userResponse
		variable /g root:maf:mafPC_Vrev1, root:maf:mafPC_Vrev2, root:maf:mafPC_isDC
		string /g root:maf:mafPC_PatternType
		PathInfo mafPCPath
		if (v_flag == 0)	// i.e. doesn't exist yet
			pathinfo Igor
			newpath /c/q/z mafPCPath, s_path + ":mafPC"
		endif
	endif
end

Window mafPC_PatternPanel() : Panel
	PauseUpdate; Silent 1		// building window...
	NewPanel /K=1 /W=(119,403,309,589) as "AuditorySet"
	ListBox patlist,pos={8,8},size={120,176},proc=PatternListboxProc
	ListBox patlist,listWave=root:maf:mafPC_Patterns,mode= 1,selRow= 1
	Button EditButton,pos={136,74},size={50,20},proc=PatternListButtonProc,title="Edit"
	Button EditButton,help={"Edit this pattern"}
	Button NewButton,pos={136,8},size={50,20},proc=PatternListButtonProc
	Button NewButton,help={"Create new pattern"}
	Button DeleteButton,pos={136,119},size={50,20},proc=PatternListButtonProc,title="Delete"
	Button DeleteButton,help={"Delete this pattern from this set"}
	Button RenameButton,pos={136,97},size={50,20},proc=PatternListButtonProc,title="Rename"
	Button RenameButton,help={"Rename this pattern"}
	Button CopyButton,pos={136,141},size={50,20},proc=PatternListButtonProc,title="Copy"
	Button CopyButton,help={"Copy this pattern"}
	Button SaveButton,pos={136,52},size={50,20},proc=PatternListButtonProc,title="Save"
	Button SaveButton,help={"Save this pattern to disk"}
	Button LoadButton,pos={136,30},size={50,20},proc=PatternListButtonProc,title="Load"
	Button LoadButton,help={"Load pattern from disk"}
	Button MoveUpButton,pos={136,163},size={22,20},proc=PatternListButtonProc,title="\\F'Symbol'"
	Button MoveUpButton,help={"Move this pattern up in the set"}
	Button MoveDownButton,pos={164,163},size={22,20},proc=PatternListButtonProc,title="\\F'Symbol'"
	Button MoveDownButton,help={"Move pattern down in the set"}
EndMacro

Function mafPC_OKCancelButtonProc(ctrlName) : ButtonControl
	String ctrlName
	NVAR mafPC_userResponse=root:maf:mafPC_userResponse
	mafPC_userResponse = (cmpstr (ctrlName, "CancelButton") == 0) ? 0 : 1
	DoWindow /K $(winname (0, 64))	// kill topmost panel
End

// prevent PauseForUser error in Windows (i.e. don't allow user to do anything if any modal dialogs are open)
function modalConflict (prefwindow, exceptwindow)
	string prefwindow, exceptwindow
	variable i
	string tempstr
	
	tempstr = "mafPC_ParamPanel;getTrainPatternPanel;mafPC_EditPanel;getNewPatternName;LoadPattern;OpenSet;getNewSetName"
	if (strlen (prefwindow) > 0)
		tempstr = prefwindow + ";" + tempstr
	endif
	tempstr = removefromlist (exceptwindow, tempstr)
	for (i = 0; i < itemsinlist (tempstr); i += 1)
		dowindow $(stringfromlist (i, tempstr))
		if (v_flag == 1)
			return 1
		endif
	endfor
	return 0
end

Function PatternListButtonProc(ctrlName) : ButtonControl
	String ctrlName
	NVAR mafPC_userResponse=root:maf:mafPC_userResponse, mafPC_sampInt=root:maf:mafPC_sampInt
	SVAR mafPC_patternname=root:maf:mafPC_patternname
	SVAR mafPC_PatternType=root:maf:mafPC_PatternType
	wave /t mafPC_Patterns=root:maf:mafPC_Patterns
	variable tempval, i
	string tempstr
	
	
	if (modalConflict ("", ""))
		return 0
	endif
	strswitch (ctrlName)
		case "NewButton":
			NewPanel /W=(371.25,485,543,574.25)
			dowindow /c getNewPatternName
			DoWindow /T getNewPatternName, "New Pattern"
			SetDrawLayer UserBack
			DrawText 7,23,"Enter the new pattern name:"
			SetVariable setvar0,pos={17,26},size={123,19},title=" ",fSize=12
			SetVariable setvar0,value= root:maf:mafPC_patternName
			Button NewButton,pos={106,68},size={50,20},proc=mafPC_OKCancelButtonProc
			Button CancelButton,pos={45,68},size={50,20},proc=mafPC_OKCancelButtonProc,title="Cancel"
			CheckBox IntervalButton,pos={24,50},size={53,14},title="Interval",value= 1,mode=1,proc=PatternTypeCheckProc
			CheckBox EventButton,pos={88,50},size={46,14},title="Event",value= 0,mode=1,proc=PatternTypeCheckProc
			mafPC_patternName = ""
			mafPC_PatternType = "Interval"	// default value
			autopositionwindow /m=1/r=mafPC_PatternPanel
			controlupdate setvar0 
			SetVariable setvar0 activate
			pauseforuser getNewPatternName
			if (mafPC_userResponse == 1 && strlen (mafPC_patternname) > 0 && findwavestrval (mafPC_Patterns, mafPC_patternname) == -1)	// i.e. OK
				redimension /n=(numpnts (mafpc_patterns) + 1) mafPC_Patterns
				mafPC_Patterns[numpnts (mafpc_patterns) - 1] = mafPC_patternname
				string /g $("root:maf:" + mafPC_Patternname)
				SVAR currPat = $("root:maf:" + mafPC_Patternname)
				mafPC_sampInt = 0
				currPat = "NUMROWS:0;SAMPINT:10;CONTROL:;SAVECHAN:;PATTYPE:" + mafPC_PatternType
				listbox patlist selrow=numpnts (mafpc_patterns) - 1
				patternlistboxproc ("", numpnts (mafpc_patterns) - 1, 0, 0)
				mafPC_EditPattern ()
				if (mafPC_userResponse == 0 || numberbykey ("NUMROWS", currPat) == 0)	// cancelled or empty
					killstrings currPat
					deletepoints numpnts (mafpc_patterns) - 1, 1, mafPC_Patterns
					listbox patlist selrow=-1
					patternlistboxproc ("", -1, -1, 0)
				endif
			endif
			break
		case "EditButton":
			tempval = checkval ("patlist", "")
			if (tempval < 0 || tempval > numpnts (mafPC_Patterns) - 1)
				break
			endif
			mafPC_patternname = mafPC_Patterns[tempval]
			mafPC_EditPattern ()
			SVAR currPat = $("root:maf:" + mafPC_patternname)
			if (numberbykey ("NUMROWS", currPat) == 0)	// was pattern eviscerated?
				killstrings currPat
				deletepoints tempval, 1, mafPC_Patterns
				listbox patlist selrow=-1
				patternlistboxproc ("", -1, -1, 0)
			endif
			break
		case "RenameButton":
			tempval = checkval ("patlist", "")
			if (tempval < 0 || tempval > numpnts (mafPC_Patterns) - 1)
				break
			endif
			NewPanel /W=(371.25,485,543,574.25)
			dowindow /c getNewPatternName
			DoWindow /T getNewPatternName, "Rename Pattern"
			SetDrawLayer UserBack
			DrawText 7,23,"Enter the new pattern name:"
			SetVariable setvar0,pos={39,26},size={123,19},title=" ",fSize=12
			SetVariable setvar0,value= root:maf:mafPC_patternName
			mafPC_PatternName = mafPC_Patterns[tempval]	// set to current name
			Button RenameButton,pos={112,57},size={50,20},proc=mafPC_OKCancelButtonProc,title="Rename"
			Button CancelButton,pos={51,57},size={50,20},proc=mafPC_OKCancelButtonProc,title="Cancel"
			autopositionwindow /m=1/r=mafPC_PatternPanel
			controlupdate setvar0 
			setvariable setvar0, activate
			pauseforuser getNewPatternName
			if (mafPC_userResponse == 1 && strlen (mafPC_patternname) > 0 && findwavestrval (mafPC_Patterns, mafPC_patternname) == -1)	// i.e. OK
				rename $("root:maf:" + mafPC_Patterns[tempval]) $(mafPC_Patternname)
				mafPC_Patterns[tempval] = mafPC_PatternName
			endif
			SVAR currPat = $("root:maf:" + mafPC_patternname)
			currPat = replacenumberbykey ("COMPILED", currPat, 0)
			break
			
		case "DeleteButton":
			tempval = checkval ("patlist", "")
			if (tempval < 0 || tempval > numpnts (mafPC_Patterns) - 1)
				break
			endif
			SVAR currPat = $("root:maf:" + mafPC_Patterns[tempval])
			killstrings currPat
			deletepoints tempval, 1, mafPC_Patterns
			listbox patlist selrow=-1
			patternlistboxproc ("", -1, -1, 0)
			break
		case "CopyButton":
			tempval = checkval ("patlist", "")
			if (tempval < 0 || tempval > numpnts (mafPC_Patterns) - 1)
				break
			endif
			string /g $("root:maf:" + mafPC_Patterns[tempval] + "Copy")
			SVAR currPat = $("root:maf:" + mafPC_Patterns[tempval]), copyPat=$("root:maf:" + mafPC_Patterns[tempval] + "Copy")
			copyPat = replacenumberbykey ("COMPILED", currPat, 0)
			insertpoints tempval + 1, 1, mafPC_Patterns
			mafPC_Patterns[tempval + 1] = mafPC_Patterns[tempval] + "Copy"
			break
		case "SaveButton":
			tempval = checkval ("patlist", "")
			if (tempval >= 0 && tempval < numpnts (mafPC_Patterns))
				mafPC_SavePattern (mafPC_Patterns[tempval])
			endif
			break
		case "LoadButton":
			mafPC_LoadPattern ("")
			break
		case "MoveUpButton":
			tempval = checkval ("patlist", "")
			if (tempval < 1 || tempval > numpnts (mafPC_Patterns) - 1 || numpnts (mafPC_Patterns) == 1)
				break
			endif
			tempstr = mafPC_Patterns[tempval]
			mafPC_Patterns[tempval] = mafPC_Patterns[tempval - 1]
			mafPC_Patterns[tempval - 1] = tempstr
			listbox patlist selrow=tempval - 1
			if (tempval == 1)
				button MoveUpButton disable = 2
			endif
			button MoveDownButton disable = 0
			break
		case "MoveDownButton":
			tempval = checkval ("patlist", "")
			if (tempval < 0 || tempval > numpnts (mafPC_Patterns) - 2 || numpnts (mafPC_Patterns) == 1)
				break
			endif
			tempstr = mafPC_Patterns[tempval]
			mafPC_Patterns[tempval] = mafPC_Patterns[tempval + 1]
			mafPC_Patterns[tempval + 1] = tempstr
			listbox patlist selrow=tempval + 1
			if (tempval == numpnts (mafPC_Patterns) - 2)
				button MoveDownButton disable = 2
			endif
			button MoveUpButton disable=0
			break
	endswitch
End
function PatternListboxProc (ctrlName, row, col, event)
	string ctrlName
	variable row, col, event
	wave /t mafPC_Patterns=root:maf:mafPC_Patterns
	variable d
	
	d = (row >= 0 && row < numpnts (mafPC_Patterns)) ? 0 : 2
	button EditButton disable=d
	button RenameButton disable=d
	button DeleteButton disable=d
	button CopyButton disable=d
	button SaveButton disable=d
	button MoveUpButton disable=d
	button MoveDownButton disable=d
	if (row == 0)
		button MoveUpButton disable=2
	endif
	if (row == numpnts (mafPC_Patterns) - 1)
		button MoveDownButton disable=2
	endif
	return 0
end
Function PatternTypeCheckProc(ctrlName,checked) : CheckBoxControl
	String ctrlName
	Variable checked
	SVAR mafPC_PatternType=root:maf:mafPC_PatternType
	strswitch (ctrlName)
		case "IntervalButton":
			checkbox IntervalButton value=1
			checkbox EventButton value=0
			mafPC_PatternType = "Interval"
			break
		case "EventButton":
			checkbox IntervalButton value=0
			checkbox EventButton value=1
			mafPC_PatternType = "Event"
			break
	endswitch
end

Function EditOKCancelButtonProc(ctrlName) : ButtonControl
	String ctrlName
	NVAR mafPC_userResponse=root:maf:mafPC_userResponse
	mafPC_userResponse = (cmpstr (ctrlName, "OKButton") == 0) ? 1 : 0
	DoWindow /K mafPC_EditPanel
end
function numberByKeyOrDefault (varname, liststr, value)
	string varname, liststr
	variable value
	variable tempval
	tempval = numberbykey (varname, liststr)
	return (numtype (tempval) == 2 ? value : tempval)
end
function updateEditListLabels ()
	SVAR mafPC_control=root:maf:mafPC_control
	wave /t mafPC_PatternSpecs=root:maf:mafPC_PatternSpecs
	variable i, totalwidth, countcols
	string widthStr

	setdimlabel 1, 0, Step, mafPC_PatternSpecs
	setdimlabel 1, 1, $("Drtn (ms)"), mafPC_PatternSpecs
	widthstr = "30,50"
	totalwidth = 80
	countcols = 2
	for (i = 0; i < itemsinlist (mafPC_control); i += 1)
		if (cmpstr ((stringfromlist (i, mafPC_control))[0,1], "DA") == 0)
			setdimlabel 1, countcols, $(stringfromlist (i, mafPC_control)), mafPC_PatternSpecs
			setdimlabel 1, countcols + 1, Abs, mafPC_PatternSpecs
			widthstr += ",40,25"
			totalwidth += 65
			countcols += 2
		elseif (cmpstr (stringfromlist (i, mafPC_control), "TTL") == 0)
			setdimlabel 1, countcols, TTL0, mafPC_PatternSpecs
			setdimlabel 1, countcols + 1, '1', mafPC_PatternSpecs
			setdimlabel 1, countcols + 2, '2', mafPC_PatternSpecs
			setdimlabel 1, countcols + 3, '3', mafPC_PatternSpecs
			widthstr += ",35,20,20,20"
			totalwidth += 35 + 20*3
			countcols += 4
		endif
	endfor
	execute "Listbox editList, widths={" + widthstr + "}"
	listbox editlist, size={totalwidth + 21, 208}	// widen listbox to accommodate more columns - should also widen window
end
function hasTTL (chlist)
	string chlist
	return (findlistitem ("TTL", chlist) != -1)
end
function countDA (chlist)
	string chlist
	variable i, numDA = 0
	for (i = 0; i < itemsinlist (chlist); i += 1)
		if (cmpstr ((stringfromlist (i, chlist))[0,1], "DA") == 0)
			numDA += 1
		endif
	endfor
	return (numDA)
end
function /t thisChanNum (whichItem, chlist)
	variable whichItem
	string chlist
	return ((stringfromlist (whichItem, chlist))[2])
end
function /t addListItemIfNotThere (newItem, listStr, sepStr)
	string newItem, listStr, sepStr
	return (selectstring (findListItem (newItem, listStr, sepStr) == -1, listStr, addlistitem (newItem, listStr, sepStr)))
end
function mafPC_EditPattern ()
	variable numrows, i, j, numDA, doTTL, bits
	string istr, ch
	SVAR mafPC_PatternName=root:maf:mafPC_PatternName, currPat=$("root:maf:" + mafPC_PatternName)
	wave /t mafPC_Patterns=root:maf:mafPC_Patterns
	NVAR mafPC_sampInt=root:maf:mafPC_sampInt, mafPC_userResponse=root:maf:mafPC_userResponse
	NVAR mafPC_isDC=root:maf:mafPC_isDC
	SVAR mafPC_control=root:maf:mafPC_control, mafPC_saveChan=root:maf:mafPC_saveChan
	SVAR mafPC_mini1=root:maf:mafPC_mini1,mafPC_mini2=root:maf:mafPC_mini2
	NVAR mafPC_vrev1=root:maf:mafPC_vrev1, mafPC_vrev2=root:maf:mafPC_vrev2
	
	numrows = numberbykeyordefault ("NUMROWS", currPat, 0)
	mafPC_sampInt = numberbykeyordefault ("SAMPINT", currPat, 10)
	mafPC_saveChan = sortlist (replacestring (",", stringbykey ("SAVECHAN", currPat), ";"))
	mafPC_control = sortlist (replacestring (",", stringbykey ("CONTROL", currPat), ";"))
	if (cmpstr (stringbykey ("PATTYPE", currPat), "Interval") == 0)
		// translate each row into specs and selection
		numDA = countDA (mafPC_control)
		doTTL = hasTTL (mafPC_control)
		make /o/n=(numrows, 2 + numDA * 2 + doTTL * 4)/t root:maf:mafPC_PatternSpecs
		make /o/n=(numrows, 2 + numDA * 2 + doTTL * 4)/B/U root:maf:mafPC_PatternSel
		make /o/n=(numrows)/t root:maf:mafPC_PatternParams
		wave /t mafPC_PatternSpecs=root:maf:mafPC_PatternSpecs, mafPC_PatternParams=root:maf:mafPC_PatternParams
		wave mafPC_PatternSel=root:maf:mafPC_PatternSel
	
		// read in starting values
		for (i = 0; i < numrows; i += 1)
			istr = num2str (i)
			mafPC_PatternParams[i] = ""
			mafPC_PatternSpecs[i][0] = istr
			mafPC_PatternSel[i][0] = 0
			mafPC_PatternSpecs[i][1] = stringbykey ("TIME" + istr, currPat)
			mafPC_PatternSel[i][1] = 2
			if (cmpstr (mafPC_PatternSpecs[i][1], "TRAIN") == 0)
				// if train, just clear rest of line (really should disable)
				for (j = 0; j < numDA; j += 1)
					mafPC_PatternSpecs[i][2 + j * 2] = ""
					mafPC_PatternSel[i][2 + j * 2] = 2
					mafPC_PatternSpecs[i][3 + j * 2] = ""
					mafPC_PatternSel[i][3 + j * 2] = 0x20
				endfor
				if (doTTL)
					for (j = 0; j < 4; j += 1)
						mafPC_PatternSpecs[i][2 + numDA * 2, 5 + numDA * 2] = ""
						mafPC_PatternSel[i][2 + numDA * 2, 5 + numDA * 2] = 0x20
					endfor
				endif
				mafPC_PatternParams[i] = "PATTERN:" + stringbykey ("PATTERN" + istr, currPat)
				mafPC_PatternParams[i] += ";REP:" + stringbykey ("REP" + istr, currPat)
			else
				// get DA amps
				for (j = 0; j < numDA; j += 1)
					ch = thischannum (j, mafPC_control)
					mafPC_PatternSpecs[i][2 + j * 2] = stringbykey ("AMP" + istr + "_" + ch, currPat)
					mafPC_PatternSel[i][2 + j * 2] = 2
					if (cmpstr (mafPC_PatternSpecs[i][2 + j * 2], "RAMP") == 0)
						mafPC_PatternParams[i] = replacestringbykey ("RAMPSTART" + ch, mafPC_PatternParams[i], stringbykey ("RAMPSTART" + istr + "_" + ch, currPat))
						mafPC_PatternParams[i] = replacestringbykey ("RAMPEND" + ch, mafPC_PatternParams[i], stringbykey ("RAMPEND" + istr + "_" + ch, currPat))
					elseif (cmpstr (mafPC_PatternSpecs[i][2 + j * 2], "WAVE") == 0)
						mafPC_PatternParams[i] = replacestringbykey ("WAVE" + ch, mafPC_PatternParams[i], stringbykey ("WAVE" + istr + "_" + ch, currPat))
					endif
					mafPC_PatternSpecs[i][3 + j * 2] = ""
					mafPC_PatternSel[i][3 + j * 2] = 0x20 + 0x10 * numberbykey ("ABS" + istr + "_" + ch, currPat)
				endfor
				if (doTTL) // set TTLs
					for (j = 0; j < 4; j += 1)
						mafPC_PatternSpecs[i][2 + numDA * 2 + j] = ""
						mafPC_PatternSel[i][2 + numDA * 2 + j] = 0x20 + ((numberbykey ("TTL" + istr, currPat) & 2^j) ? 0x10 : 0)
					endfor
				endif
			endif
		 endfor
	
		// put up panel
		NewPanel /W=(866.25,49.25,1230.75,317.75) as "Edit Interval: " + mafPC_PatternName
		SetVariable setvar001,pos={6,7},size={186,19},title="Sampling Interval (s):"
		SetVariable setvar001,fSize=12,limits={0,Inf,0},value= root:maf:mafPC_sampInt
		PopupMenu ADpopup,pos={221,5},size={59,21},proc=ADPopMenuProc,title="Record"
		PopupMenu ADpopup,fSize=12,mode=0,value= #"ADPopVal(root:maf:mafPC_saveChan, \"AD0;AD1;AD2;AD3;AD4;AD5;AD6;AD7;TTL\")"
		PopupMenu DApopup,pos={276,5},size={59,21},proc=IntervalDAPopMenuProc,title="Control"
		PopupMenu DApopup,mode=0,value= #"ADPopVal(root:maf:mafPC_control, \"DA0;DA1;DA2;DA3;TTL\")"
		Button Delbutton,pos={11,245},size={40,20},proc=IntervalButtonsProc,title="Delete"
		Button Insertbutton,pos={53,245},size={40,20},proc=IntervalButtonsProc,title="Insert"
		Button Trainbutton,pos={96,245},size={40,20},disable=2,proc=IntervalButtonsProc,title="Train"
		Button Cancelbutton,pos={267,245},size={40,20},proc=mafPC_OKCancelButtonProc,title="Cancel"
		Button OKbutton,pos={310,245},size={40,20},proc=mafPC_OKCancelButtonProc,title="OK"
		Button Rampbutton,pos={138,245},size={40,20},disable=2,proc=IntervalButtonsProc,title="Ramp"
		Button Wavebutton,pos={181,245},size={40,20},disable=2,proc=IntervalButtonsProc,title="Wave"
		ListBox editList,pos={10,30},size={252,208},proc=editlistboxproc,frame=3
		ListBox editList,listWave=root:maf:mafPC_PatternSpecs,selWave=root:maf:mafPC_PatternSel,mode= 6
		
		dowindow /c mafPC_EditPanel
		autopositionwindow /m=1/r=mafPC_PatternPanel
		updateEditListLabels ()
		pauseforuser mafPC_EditPanel
		if (mafPC_userResponse != 0)	// didn't cancel
			// update pattern based on PatternSpecs & co.
			currPat = ""
			numrows = dimsize (mafPC_PatternSpecs, 0)
			currPat = replacestringbykey ("PATTYPE", currPat, "Interval")
			currPat = replacenumberbykey ("NUMROWS", currPat, numrows)
			currPat = replacenumberbykey ("SAMPINT", currPat, mafPC_sampInt)
			currPat = replacestringbykey ("CONTROL", currPat, replacestring (";", mafPC_control, ","))
			currPat = replacestringbykey ("SAVECHAN", currPat, replacestring (";", mafPC_saveChan, ","))
			numDA = countDA (mafPC_control)
			doTTL = hasTTL (mafPC_control)
			for (i = 0; i < numrows; i += 1)
				istr = num2str (i)
				currPat = replacestringbykey ("TIME" + istr, currPat, mafPC_PatternSpecs[i][1])
				if (cmpstr (mafPC_PatternSpecs[i][1], "TRAIN") == 0)	// is it a train?
					currPat = replacestringbykey ("DEPENDENCY", currPat, AddListItemIfNotThere (stringbykey ("PATTERN", mafPC_PatternParams[i]), stringbykey ("DEPENDENCY", currPat), ","))
					currPat = replacestringbykey ("PATTERN" + istr, currPat, stringbykey ("PATTERN", mafPC_PatternParams[i]))
					currPat = replacestringbykey ("REP" + istr, currPat, stringbykey ("REP", mafPC_PatternParams[i]))
				else
					// not a train, so put in all the AMP and TTL stuff
					for (j = 0; j < numDA; j += 1)
						ch = thischannum (j, mafPC_control)
						currPat = replacestringbykey ("AMP" + istr + "_" + ch, currPat, mafPC_PatternSpecs[i][2 + j * 2])
						currPat = replacenumberbykey ("ABS" + istr + "_" + ch, currPat, (mafPC_PatternSel[i][3 + j * 2]) & 0x10 ? 1 : 0)
						if (cmpstr (mafPC_PatternSpecs[i][2 + j * 2], "RAMP") == 0)
							currPat = replacestringbykey ("RAMPSTART" + istr + "_" + ch, currPat, stringbykey ("RAMPSTART" + ch, mafPC_PatternParams[i]))
							currPat = replacestringbykey ("RAMPEND" + istr + "_" + ch, currPat, stringbykey ("RAMPEND" + ch, mafPC_PatternParams[i]))
						elseif (cmpstr (mafPC_PatternSpecs[i][2 + j * 2], "WAVE") == 0)
							currPat = replacestringbykey ("WAVE" + istr + "_" + ch, currPat, stringbykey ("WAVE" + ch, mafPC_PatternParams[i]))
							currPat = replacestringbykey ("WAVEDEPENDENCY", currPat, AddListItemIfNotThere (stringbykey ("WAVE" + ch, mafPC_PatternParams[i]), stringbykey ("WAVEDEPENDENCY", currPat), ","))
						endif
					endfor
					if (doTTL)
						bits = 0
						for (j = 0; j < 4; j += 1)
							bits += (mafPC_PatternSel[i][2 + numDA * 2 + j] & 0x10) ? 2^j : 0
						endfor
						currPat = replacenumberbykey ("TTL" + istr, currPat, bits)
					endif
				endif
			endfor
		endif
	elseif (cmpstr (stringbykey ("PATTYPE", currPat), "Event") == 0)
		// setup waves
		make /o/t/n=(numrows, 5) root:maf:mafPC_PatternSpecs
		make /o/t/n=(numrows) root:maf:mafPC_PatternParams
		make /o/n=(numrows, 5) root:maf:mafPC_PatternSel
		wave /t mafPC_PatternSpecs=root:maf:mafPC_PatternSpecs, mafPC_PatternParams=root:maf:mafPC_PatternParams
		wave mafPC_PatternSel=root:maf:mafPC_PatternSel
		setdimlabel 1, 0, $("Time (ms)"), mafPC_PatternSpecs
		setdimlabel 1, 1, Event, mafPC_PatternSpecs
		setdimlabel 1, 2, $("Ch/Bit"), mafPC_PatternSpecs
		setdimlabel 1, 3, Value, mafPC_PatternSpecs
		setdimlabel 1, 4, Abs, mafPC_PatternSpecs
		mafPC_control = removefromlist ("G1;G2;GForce", mafPC_control)
		for (i = 0; i < numrows; i += 1)
			istr = num2str (i)
			mafPC_PatternSpecs[i][0] = stringbykey ("TIME" + istr, currPat)
			mafPC_PatternSel[i][0] = 2
			mafPC_PatternSpecs[i][1] = stringbykey ("EVENT" + istr, currPat)
			mafPC_PatternSel[i][1] = 0
			strswitch (mafPC_PatternSpecs[i][1])
				case "TRAIN":
					mafPC_PatternSpecs[i][2] = stringbykey ("PATTERN" + istr, currPat) + ";" + stringbykey ("REP" + istr, currPat)
					mafPC_PatternSel[i][2] = 0
					mafPC_PatternSpecs[i][3,4] = ""
					mafPC_PatternSel[i][3,4] = 0
					break
				case "DCPULSE":
					mafPC_PatternSpecs[i][2] = stringbykey ("CH" + istr, currPat)
					mafPC_PatternSel[i][2] = 0
					mafPC_PatternSpecs[i][3] = stringbykey ("AMP" + istr, currPat)
					mafPC_PatternSel[i][3] = 2
					mafPC_PatternSpecs[i][4] = ""
					mafPC_PatternSel[i][4] = 0
					break
				case "AMP":
				case "RAMPSTART":
				case "RAMPEND":
					mafPC_PatternSpecs[i][2] = stringbykey ("CH" + istr, currPat)
					mafPC_PatternSel[i][2] = 0
					mafPC_PatternSpecs[i][3] = stringbykey ("AMP" + istr, currPat)
					mafPC_PatternSel[i][3] = 2
					if (cmpstr (mafPC_PatternSpecs[i][3], "WAVE") == 0)
						mafPC_PatternSpecs[i][3] += ":" + stringbykey ("WAVE" + istr, currPat)
					endif
					mafPC_PatternSpecs[i][4] = ""
					mafPC_PatternSel[i][4] = 32 + 16 * numberbykey ("ABS" + istr, currPat)
					break
				case "TTL":
					mafPC_PatternSpecs[i][2] = stringbykey ("CH" + istr, currPat)
					mafPC_PatternSel[i][2] = 0
					mafPC_PatternSpecs[i][3] = ""
					mafPC_PatternSel[i][3] = 0x20 + 0x10 * numberbykey ("AMP" + istr, currPat)
					mafPC_PatternSpecs[i][4] = ""
					mafPC_PatternSel[i][4] = 0
					break
				case "FINISH":
				case "PADFINISH":
					mafPC_PatternSpecs[i][2,4] = ""
					mafPC_PatternSel[i][2,4] = 0
					break
			endswitch
		endfor
		mafPC_isDC = numberbykeyordefault ("DYNAMICCLAMP", currPat, 0)
		if (mafPC_isDC)	// retrieve dynamic clamp parameters
			mafPC_mini1 = stringbykey ("DCMINI1", currPat)
			mafPC_mini2 = stringbykey ("DCMINI2", currPat)
			mafPC_vRev1 = numberbykeyordefault ("DCVREV1", currPat, 0)
			mafPC_vRev2 = numberbykeyordefault ("DCVREV2", currPat, 0)
		endif
		// put up panel
		NewPanel /W=(616,399,981,668) as "Edit Event: " + mafPC_PatternName
		SetVariable setvar0,pos={6,7},size={186,19},title="Sampling Interval (s):"
		SetVariable setvar0,fSize=12,limits={0,Inf,0},value= root:maf:mafPC_sampInt
		PopupMenu ADpopup,pos={203,5},size={72,21},proc=ADPopMenuProc,title="Record"
		PopupMenu ADpopup,fSize=12,mode=0,value= #"EventInputPopVal()"
		PopupMenu DApopup,pos={276,5},size={70,21},proc=EventDAPopMenuProc,title="Control"
		PopupMenu DApopup,mode=0,value= #"EventOutputPopVal()"
		Button Delbutton,pos={2,245},size={45,20},proc=EventButtonsProc,title="Delete"
		Button AddButton,pos={49,245},size={45,20},proc=EventButtonsProc,title="Add"
		Button Cancelbutton,pos={250,245},size={50,20},proc=mafPC_OKCancelButtonProc,title="Cancel"
		Button OKbutton,pos={302,245},size={50,20},proc=mafPC_OKCancelButtonProc,title="OK"
		Button SortButton,pos={150,245},size={45,20},proc=EventButtonsProc,title="Sort"
		Button MoveUpButton,pos={197,245},size={22,20},proc=EventButtonsProc,title="\\F'Symbol'"
		Button MoveUpButton,help={"Move this event up in the pattern"}
		Button MoveDownButton,pos={220,245},size={22,20},proc=EventButtonsProc,title="\\F'Symbol'"
		Button MoveDownButton,help={"Move this event down in the pattern"}
		Button WaveButton,pos={96,245},size={45,20},proc=EventButtonsProc,title="Wave"
		ListBox editList,pos={10,85},size={342,152},proc=editeventlistprocname,frame=3
		ListBox editList,listWave=root:maf:mafPC_PatternSpecs
		ListBox editList,selWave=root:maf:mafPC_PatternSel,mode= 6,widths= {50,30,30,50,20}
		SetVariable setvar1,pos={89,29},size={106,19},disable=2,title="Mini 1:",fSize=12
		SetVariable setvar1,frame=0,limits={0,Inf,0},value= root:maf:mafPC_mini1
		SetVariable setvar2,pos={195,29},size={104,19},disable=2,title="Mini 2:"
		SetVariable setvar2,fSize=12,frame=0
		SetVariable setvar2,limits={0,Inf,0},value= root:maf:mafPC_mini2
		SetVariable setvar3,pos={93,50},size={102,19},disable=2,title="Vrev (mV):"
		SetVariable setvar3,fSize=12,frame=0
		SetVariable setvar3,limits={0,Inf,0},value= root:maf:mafpc_vrev1
		SetVariable setvar4,pos={197,50},size={102,19},disable=2,title="Vrev (mV):"
		SetVariable setvar4,fSize=12,frame=0
		SetVariable setvar4,limits={0,Inf,0},value= root:maf:mafPC_Vrev2
		CheckBox DCCheck,pos={9,29},size={71,32},proc=PCDCCheckProc,title="Dynamic\rClamp"
		CheckBox DCCheck,help={"Indicate whether pattern is dynamic clamp"},fSize=12
		CheckBox DCCheck,variable=root:maf:mafPC_isDC
		GroupBox group0,pos={86,28},size={215,43},disable=2
		pcdccheckproc ("", mafPC_isDC)
		dowindow /c mafPC_EditPanel
		autopositionwindow /m=1/r=mafPC_PatternPanel
		pauseforuser mafPC_EditPanel
		
		if (mafPC_userResponse != 0)	// update pattern based on PatternSpecs & co.
			currPat = ""
			numrows = dimsize (mafPC_PatternSpecs, 0)
			currPat = replacestringbykey ("PATTYPE", currPat, "Event")
			currPat = replacenumberbykey ("NUMROWS", currPat, numrows)
			currPat = replacenumberbykey ("SAMPINT", currPat, mafPC_sampInt)
			currPat = replacestringbykey ("CONTROL", currPat, replacestring (";", mafPC_control, ","))
			currPat = replacestringbykey ("SAVECHAN", currPat, replacestring (";", mafPC_saveChan, ","))
			if (mafPC_isDC)
				currPat = replacenumberbykey ("DYNAMICCLAMP", currPat, 1)
				currPat = replacenumberbykey ("DCVREV1", currPat, mafPC_vrev1)
				currPat = replacenumberbykey ("DCVREV2", currPat, mafPC_vrev2)
				currPat = replacestringbykey ("DCMINI1", currPat, mafPC_mini1)
				currPat = replacestringbykey ("DCMINI2", currPat, mafPC_mini2)
			endif
			for (i = 0; i < numrows; i += 1)
				istr = num2str (i)
				currPat = replacestringbykey ("TIME" + istr, currPat, replacestring (";", mafPC_PatternSpecs[i][0], ","))
				currPat = replacestringbykey ("EVENT" + istr, currPat, mafPC_PatternSpecs[i][1])
				strswitch (mafPC_PatternSpecs[i][1])
					case "TRAIN":
						currPat = replacestringbykey ("PATTERN" + istr, currPat, stringfromlist (0, mafPC_PatternSpecs[i][2]))
						currPat = replacestringbykey ("REP" + istr, currPat, stringfromlist (1, mafPC_PatternSpecs[i][2]))
						currPat = replacestringbykey ("DEPENDENCY", currPat, AddListItemIfNotThere (stringfromlist (0, mafPC_PatternSpecs[i][2]), stringbykey ("DEPENDENCY", currPat), ","))
						break
					case "DCPULSE":
						currPat = replacestringbykey ("CONTROL", currPat, addlistitemIfNotThere (mafPC_PatternSpecs[i][2], stringbykey ("CONTROL", currPat), ","))
						currPat = replacestringbykey ("CH" + istr, currPat, mafPC_PatternSpecs[i][2])
						currPat = replacestringbykey ("AMP" + istr, currPat, replacestring (";", mafPC_PatternSpecs[i][3], ","))
						break
					case "AMP":
					case "RAMPSTART":
					case "RAMPEND":
						currPat = replacestringbykey ("CH" + istr, currPat, mafPC_PatternSpecs[i][2])
						currPat = replacestringbykey ("AMP" + istr, currPat, replacestring (";", mafPC_PatternSpecs[i][3], ","))
						currPat = replacenumberbykey ("ABS" + istr, currPat, (mafPC_PatternSel[i][4]) & 0x10 ? 1 : 0)
						if (haskey ("WAVE", mafPC_PatternSpecs[i][3]))
							currPat = replacestringbykey ("AMP" + istr, currPat, "WAVE")
							currPat = replacestringbykey ("WAVE" + istr, currPat, stringbykey ("WAVE", mafPC_PatternSpecs[i][3]))
							currPat = replacestringbykey ("WAVEDEPENDENCY", currPat, AddListItemIfNotThere (stringbykey ("WAVE", mafPC_PatternSpecs[i][3]), stringbykey ("WAVEDEPENDENCY", currPat), ","))
						endif
						if (mafPC_isDC && cmpstr ((mafPC_PatternSpecs[i][2])[0], "G") == 0)
							currPat = replacestringbykey ("CONTROL", currPat, addlistitemIfNotThere (mafPC_PatternSpecs[i][2], stringbykey ("CONTROL", currPat), ","))
						endif
						break
					case "TTL":
						currPat = replacestringbykey ("CH" + istr, currPat, mafPC_PatternSpecs[i][2])
						currPat = replacenumberbykey ("AMP" + istr, currPat, mafPC_PatternSel[i][3] & 0x10 ? 1 : 0)
						break
					case "FINISH":
					case "PADFINISH":
						break
				endswitch
			endfor
		endif
	endif
end

function /t ADPopVal (setthese, startstr)
	string setthese, startstr
	string returnval=startstr
	variable i, strpos
	
	for (i = 0; i < itemsinlist (setthese); i += 1)
		strpos = findlistitem (stringfromlist (i, setthese), returnval)
		if (strpos == -1)
			print "mafPC Error: Specified AD list does not match available AD channels"
		else
			returnval = returnval[0, strpos - 1] + "\M1!" + num2char (18) + returnval[strpos, inf]
		endif
	endfor
	return returnval
end

function EditListboxProc (ctrlName, row, col, event)
	string ctrlName
	variable row, col, event
	wave /t mafPC_Patterns=root:maf:mafPC_Patterns
	SVAR mafPC_control=root:maf:mafPC_control
	variable numDA = countDA (mafPC_control)
	
	button TrainButton disable=((col == 1 && numpnts (mafPC_Patterns) > 1) ? 0 : 2)
	if (numDA && col > 1 && col < numDA * 2 + 2 && mod (col, 2) == 0)	// *** should also disable if a train
		button RampButton disable=0
		button WaveButton disable=0
	else
		button RampButton disable=2
		button WaveButton disable=2
	endif
	return 0
end

Function IntervalDAPopMenuProc(ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName, popStr
	Variable popNum
	SVAR mafPC_control = root:maf:mafPC_control
	wave /t mafPC_PatternSpecs=root:maf:mafPC_PatternSpecs
	wave mafPC_PatternSel=root:maf:mafPC_PatternSel
	variable ampcol

	if (findlistitem (popstr, mafPC_control) == -1)	// not there, so add
		mafPC_control = sortlist (addlistitem (popstr, mafPC_control))
		if (dimsize (mafPC_PatternSpecs, 0) == 0)
			// can't insert columns into zero-row waves in 5.01, so need to remake
			ampcol = 2 + countDA (mafPC_control) * 2 + hasTTL (mafPC_control) * 4
			make /t/o/n=(0, ampcol) root:maf:mafPC_PatternSpecs
			make /o/n=(0, ampcol) root:maf:mafPC_PatternSel
		elseif (cmpstr (popstr, "TTL") != 0)
			ampcol = 2 + whichlistitem (popstr, mafPC_control) * 2
			insertpoints /M=1 ampcol,2,mafPC_PatternSpecs, mafPC_PatternSel
			ampcol = 2 + whichlistitem (popstr, mafPC_control) * 2
			mafPC_PatternSpecs[0, dimsize (mafPC_PatternSpecs, 0)][ampcol] = "0"
			mafPC_PatternSel[0, dimsize (mafPC_PatternSpecs, 0)][ampcol] = 2
			mafPC_PatternSpecs[0, dimsize (mafPC_PatternSpecs, 0)][ampcol + 1] = ""
			mafPC_PatternSel[0, dimsize (mafPC_PatternSpecs, 0)][ampcol + 1] = 0x20
		else		// i.e. TTL
			ampcol = dimsize (mafpc_patternspecs, 1)
			insertpoints /M=1 ampcol,4,mafPC_PatternSpecs, mafPC_PatternSel
			mafpc_patternspecs[0, dimsize (mafPC_PatternSpecs, 0)][ampcol,ampcol + 3] = ""
			mafpc_patternsel[0, dimsize (mafPC_PatternSpecs, 0)][ampcol,ampcol + 3] = 0x20
		endif
		// initialize values
	else	// there already, so need to kill relevant columns
		if (cmpstr (popstr, "TTL") != 0)
			ampcol = 2 + whichlistitem (popstr, mafPC_control) * 2
			DeletePoints /M=1 ampcol,2,mafPC_PatternSpecs, mafPC_PatternSel
		else
			ampcol = 2 + countDA (mafPC_control) * 2
			DeletePoints /M=1 ampcol,4,mafPC_PatternSpecs, mafPC_PatternSel
		endif
		mafPC_control = sortlist (removefromlist (popstr, mafPC_control))
	endif
	updateEditListLabels ()
End

function findSelectedRow (w)
	wave w
	variable i, j
	if (dimsize (w, 1) > 0)
		for (i = 0; i < dimsize (w, 0); i += 1)
			for (j = 0; j < dimsize (w, 1); j += 1)
				if (w[i][j] & 0x01)	// is this cell selected?
					return i
				endif
			endfor
		endfor
	else
		for (i = 0; i < numpnts (w); i += 1)
			if (w[i] & 0x01)	// is this cell selected?
				return i
			endif
		endfor
	endif
	return -1
end
function findSelectedCol (w)
	wave w
	variable i, j
	if (dimsize (w, 1) == 0)
		return 0
	endif
	for (i = 0; i < dimsize (w, 0); i += 1)
		for (j = 0; j < dimsize (w, 1); j += 1)
			if (w[i][j] & 0x01)	// is this cell selected?
				return j
			endif
		endfor
	endfor
	return -1
end

Function IntervalButtonsProc(ctrlName) : ButtonControl
	String ctrlName
	SVAR mafPC_param1=root:maf:mafPC_param1, mafPC_param2=root:maf:mafPC_param2
	SVAR mafPC_control=root:maf:mafPC_control
	NVAR mafPC_userResponse=root:maf:mafPC_userResponse
	wave mafPC_PatternSel=root:maf:mafPC_PatternSel
	wave /t mafPC_PatternSpecs=root:maf:mafPC_PatternSpecs, mafPC_PatternParams=root:maf:mafPC_PatternParams
	variable thisrow, i, numDA, thiscol
	string ch

	thisrow = findSelectedRow (mafPC_PatternSel)
	numDA = countDA (mafPC_control)
	strswitch (ctrlName)
		case "InsertButton":
			if (thisrow == -1)	// no row specified, so add at end
				thisrow = dimsize (mafPC_PatternSpecs, 0) - 1
			endif
			InsertPoints /M=0 thisrow, 1, mafPC_PatternSel, mafPC_PatternSpecs, mafPC_PatternParams
			// update step numbers
			for (i = 0; i < dimsize (mafPC_PatternSpecs, 0); i += 1)
				mafPC_Patternspecs[i][0] = num2str (i)
			endfor
			// copy values into new row
			if (dimsize (mafPC_PatternSpecs, 0) == 1)	// i.e. did we just insert the first row?
				// initialize first row
				mafPC_PatternSel[0][0] = 0
				mafPC_PatternSpecs[0][1] = "0"
				mafPC_PatternSel[0][1] = 2
				for (i= 0; i < numDA; i += 1)
					mafPC_PatternSpecs[0][2 + i * 2] = "0"
					mafPC_PatternSel[0][2 + i * 2] = 2
					mafPC_PatternSpecs[0][3 + i * 2] = ""
					mafPC_PatternSel[0][3 + i * 2] = 32
				endfor
				if (hasTTL (mafPC_control))
					for (i = 0; i < 4; i += 1)
						mafPC_PatternSpecs[0][2 + numDA * 2 + i] = ""
						mafPC_PatternSel[0][2 + numDA * 2 + i] = 32
					endfor
				endif
				mafPC_PatternParams = ""
			else
				// duplicate values into new row
				for (i = 1; i < dimsize (mafPC_PatternSpecs, 1); i += 1)
					mafPC_PatternSpecs[thisrow][i]=mafPC_PatternSpecs[thisrow + 1][i]
					mafPC_PatternSel[thisrow][i]=mafPC_PatternSel[thisrow + 1][i]
					mafPC_PatternSel[thisrow + 1][i] = mafPC_PatternSel[thisrow + 1][i] & ~0x01	// make sure selection does not get extended
				endfor
				mafPC_PatternParams[thisrow] = mafPC_PatternParams[thisrow + 1]
			endif
			Button Delbutton disable=0	// enable delete button
			break
		case "DelButton":
			if (dimsize (mafPC_patternspecs, 0) == 1 || thisrow == -1)	// nothing to delete
				break
			endif
			// update listbox
			deletePoints /M=0 thisrow, 1, mafPC_PatternSel, mafPC_PatternSpecs, mafPC_PatternParams
			for (i = 0; i < dimsize (mafPC_PatternSpecs, 0); i += 1)	// fix step numbers
				mafPC_PatternSpecs[i][0] = num2str (i)
			endfor
			if (dimsize (mafPC_patternspecs, 0) == 0)	// no rows left, so disable delete button
				Button Delbutton disable=2
			endif
			break
		case "TrainButton":
			if (dimsize (mafPC_patternspecs, 0) == 0 || thisrow == -1)
				return 0
			endif
			// get current train values, if any, and put into train dialog fields: param1 = name, param2 = reps
			if (cmpstr (mafPC_PatternSpecs[thisrow][1], "TRAIN") == 0)		// already a train
				mafPC_param1 = stringbykey ("PATTERN", mafPC_PatternParams[thisrow])
				mafPC_param2 = stringbykey ("REP", mafPC_PatternParams[thisrow])
			else
				mafPC_param1 = ""
				mafPC_param2 ="1"
			endif
			// bring up box
			NewPanel /W=(840.75,455.75,1037.25,542.75) as "Train"
			SetVariable setvar001,pos={11,31},size={153,19},proc=TrainRepsProc,title="# repetitions:"
			SetVariable setvar001,fSize=12,value= root:maf:mafPC_param2
			Button OKbutton,pos={87,61},size={50,20},proc=TrainOKCancelButtonProc,title="OK"
			Button Cancelbutton,pos={30,61},size={50,20},proc=TrainOKCancelButtonProc,title="Cancel"
			PopupMenu popup0,pos={0,6},size={150,24},title="Pattern name:",fSize=12
			PopupMenu popup0,mode=1,popvalue="",value= #"patternlist()"
			dowindow /c getTrainPatternPanel
			autopositionwindow /m=1/r=mafPC_EditPanel
			// set popup to previous value (if any)
			popupmenu Popup0 mode=(max (whichListItem (mafPC_param1, patternlist())+1, 1))
			pauseforuser getTrainPatternPanel
			if (mafPC_userResponse == 0)	// i.e. Cancel
				break
			endif
			// update listbox -- clear amps and TTLs
			mafPC_PatternSpecs[thisrow][1] = "TRAIN"
			for (i = 0; i < numDA; i += 1)
				mafPC_PatternSpecs[thisrow][2 + i * 2] = ""
				mafPC_PatternSel[thisrow][3 + i * 2] = 32		
			endfor
			if (hasTTL (mafPC_control))
				for (i = 0; i < 4; i += 1)
					mafPC_PatternSpecs[thisrow][2 + numDA * 2 + i] = ""
					mafPC_PatternSel[thisrow][2 + numDA * 2 + i] = 32
				endfor
			endif
			mafPC_PatternParams[thisrow] = "PATTERN:" + mafPC_param1
			mafPC_PatternParams[thisrow] += ";REP:" + mafPC_param2
			break
		case "RampButton":
			thiscol = findSelectedCol (mafPC_PatternSel)
			if (dimsize (mafPC_patternspecs, 0) == 0 || thisrow == -1)
				break
			endif
			if (cmpstr (mafPC_PatternSpecs[thisrow][1], "TRAIN") == 0)
				break
			endif
			ch = thisChanNum ((thiscol - 2) / 2, mafPC_control)
			// get current ramp values, if any, and put into train dialog fields
			if (cmpstr (mafPC_PatternSpecs[thisrow][thiscol], "RAMP") == 0)		// already a ramp
				mafPC_param1 = stringbykey ("RAMPSTART" + ch, mafPC_PatternParams[thisrow])
				mafPC_param2 = stringbykey ("RAMPEND" + ch, mafPC_PatternParams[thisrow])
			else
				mafPC_param1 = "0"	// initialize to last amp?
				mafPC_param2="0"
			endif
			// bring up box
			NewPanel /W=(693,503,889,590) as "Ramp"
			SetVariable setvar0,pos={11,11},size={153,16},proc=TrainRepsProc,title="Start Amp:"
			SetVariable setvar0,fSize=12,value= root:maf:mafPC_param1
			Button OKbutton,pos={87,61},size={50,20},proc=mafPC_OKCancelButtonProc,title="OK"
			Button Cancelbutton,pos={30,61},size={50,20},proc=mafPC_OKCancelButtonProc,title="Cancel"
			SetVariable setvar1,pos={14,33},size={150,16},proc=TrainRepsProc,title="End Amp:"
			SetVariable setvar1,fSize=12,value= root:maf:mafPC_param2
			dowindow /c getRampPatternPanel
			autopositionwindow /m=1/r=mafPC_EditPanel
			pauseforuser getRampPatternPanel
			if (mafPC_userResponse == 0)	// i.e. Cancel
				break
			endif
			mafPC_PatternSpecs[thisrow][thiscol] = "RAMP"
			mafPC_PatternParams[thisrow] = replacestringbykey ("RAMPSTART" + ch, mafPC_PatternParams[thisrow], mafPC_param1)
			mafPC_PatternParams[thisrow] = replacestringbykey ("RAMPEND" + ch, mafPC_PatternParams[thisrow], mafPC_param2)
			break
		case "WaveButton":
			thiscol = findSelectedCol (mafPC_PatternSel)
			if (dimsize (mafPC_patternspecs, 0) == 0 || thisrow == -1)
				break
			endif
			if (cmpstr (mafPC_PatternSpecs[thisrow][1], "TRAIN") == 0)
				break
			endif
			ch = thisChanNum ((thiscol - 2) / 2, mafPC_control)
			// get current wave values, if any, and put into dialog field
			if (cmpstr (mafPC_PatternSpecs[thisrow][thiscol], "WAVE") == 0)		// already a wave
				mafPC_param1 = stringbykey ("WAVE" + ch, mafPC_PatternParams[thisrow])
			else
				mafPC_param1 = ""	// initialize to last amp?
			endif
			// bring up box
			NewPanel /W=(693,503,889,590) as "Wave"
			SetVariable setvar0,pos={11,11},size={153,16},proc=TrainRepsProc,title="Wave name:"
			SetVariable setvar0,fSize=12,value= root:maf:mafPC_param1
			Button OKbutton,pos={87,61},size={50,20},proc=mafPC_OKCancelButtonProc,title="OK"
			Button Cancelbutton,pos={30,61},size={50,20},proc=mafPC_OKCancelButtonProc,title="Cancel"
			dowindow /c getWavePatternPanel
			autopositionwindow /m=1/r=mafPC_EditPanel
			controlupdate setvar0 
			SetVariable setvar0 activate
			pauseforuser getWavePatternPanel
			if (mafPC_userResponse == 0 || strlen (mafpc_param1) == 0)	// i.e. Cancel
				break
			endif
			mafPC_PatternSpecs[thisrow][thiscol] = "WAVE"
			mafPC_PatternParams[thisrow] = replacestringbykey ("WAVE" + ch, mafPC_PatternParams[thisrow], mafPC_param1)
			break
	endswitch
End

Function TrainRepsProc(ctrlName,varNum,varStr,varName) : SetVariableControl
	String ctrlName, varStr, varName
	Variable varNum
	Button OKbutton disable=((strlen (varstr) == 0) ? 2 : 0)
End
Function TrainOKCancelButtonProc(ctrlName) : ButtonControl
	String ctrlName
	NVAR mafPC_userResponse=root:maf:mafPC_userResponse
	SVAR mafPC_param1=root:maf:mafPC_param1
	mafPC_userResponse = (cmpstr (ctrlName, "OKButton") == 0)
	mafPC_param1 = popupval ("Popup0", "")
	DoWindow /K gettrainpatternpanel
end
function /t PatternList()
	SVAR mafPC_PatternName=root:maf:mafPC_PatternName
	wave /t mafPC_Patterns=root:maf:mafPC_Patterns
	variable i
	string returnstring=""
	for (i = 0; i < numpnts (mafPC_Patterns); i += 1)
		if (cmpstr (mafPC_Patterns[i], mafPC_PatternName) != 0)
			returnstring += mafPC_Patterns[i] + ";"
		endif
	endfor
	return (returnstring)
end
function /t EventOutputPopVal ()
	SVAR mafPC_control = root:maf:mafPC_control
	NVAR mafPC_isDC = root:maf:mafPC_isDC
	string returnval="ack"
	variable i, strpos
	if (mafPC_isDC)
		returnval = "DA1;DA2;TTL"
	else
		returnval = "DA0;DA1;DA2;DA3;TTL"
	endif
	for (i = 0; i < itemsinlist (mafPC_control); i += 1)
		strpos = findlistitem (stringfromlist (i, mafPC_control), returnval)
		if (strpos == -1)
			print "mafPC Error: Specified DA list does not match available DA channels"
		else
			returnval = returnval[0, strpos - 1] + "\M1!" + num2char (18) + returnval[strpos, inf]
		endif
	endfor
	return returnval
end
function /t EventInputPopVal ()
	SVAR mafPC_saveChan = root:maf:mafPC_saveChan
	NVAR mafPC_isDC = root:maf:mafPC_isDC
	string returnval="ack"
	variable i, strpos
	if (mafPC_isDC)
		returnval = "AD0;AD1;AD2;AD3;TTL"
	else
		returnval = "AD0;AD1;AD2;AD3;AD4;AD5;AD6;AD7;TTL"
	endif
	for (i = 0; i < itemsinlist (mafPC_saveChan); i += 1)
		strpos = findlistitem (stringfromlist (i, mafPC_saveChan), returnval)
		if (strpos == -1)
			print "mafPC Error: Specified AD list does not match available AD channels"
		else
			returnval = returnval[0, strpos - 1] + "\M1!" + num2char (18) + returnval[strpos, inf]
		endif
	endfor
	return returnval
end
Function ADPopMenuProc(ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName, popStr
	Variable popNum
	SVAR mafPC_saveChan = root:maf:mafPC_saveChan
	if (findlistitem (popstr, mafPC_saveChan) == -1)	// not there, so add
		mafPC_saveChan = addlistitem (popstr, mafPC_saveChan)
	else
		mafPC_saveChan = removefromlist (popstr, mafPC_saveChan)
	endif
	mafPC_saveChan = sortlist (mafPC_saveChan)
End
Function EventDAPopMenuProc(ctrlName,popNum,popStr) : PopupMenuControl
	String ctrlName, popStr
	Variable popNum
	SVAR mafPC_control = root:maf:mafPC_control
	if (findlistitem (popstr, mafPC_control) == -1)	// not there, so add
		mafPC_control = addlistitem (popstr, mafPC_control)
	else
		mafPC_control = removefromlist (popstr, mafPC_control)
	endif
	mafPC_control = sortlist (mafPC_control)
End
Function PCDCCheckProc(ctrlName,checked) : CheckBoxControl
	String ctrlName
	Variable checked
	variable d = 2 * (1 - checked)
	setvariable setvar1 disable=d, frame=checked
	setvariable setvar2 disable=d, frame=checked
	setvariable setvar3 disable=d, frame=checked
	setvariable setvar4 disable=d, frame=checked
	groupbox group0 disable=d
End
Function EditEventListProcName(ev)
	STRUCT WMListboxAction &ev
	wave /t mafpc_patternspecs=root:maf:mafpc_patternspecs	, mafPC_PatternParams=root:maf:mafPC_PatternParams
	wave mafPC_PatternSel=root:maf:mafPC_PatternSel
	SVAR mafPC_param1=root:maf:mafPC_param1, mafPC_param2=root:maf:mafPC_param2
	SVAR mafPC_control = root:maf:mafPC_control
	NVAR mafPC_userResponse=root:maf:mafPC_userResponse, mafPC_isDC=root:maf:mafPC_isDC
	string popupstr = ""
	variable d
	
	if (modalConflict ("", "mafPC_EditPanel"))
		return 0
	endif
	d = (ev.row >= 0 && ev.row < dimsize (mafPC_PatternSpecs, 0)) ? 0 : 2
	button DelButton disable=d
	button MoveUpButton disable=(ev.row == 0 ? 2 : d)
	button MoveDownButton disable=((ev.row == dimsize (mafPC_PatternSpecs, 0) - 1) ? 2 : d)
	button WaveButton disable=((ev.col == 3 && mafpc_patternsel[ev.row][ev.col] & 2) ? d : 2)

	if ((ev.eventcode == 1 || ev.eventcode2 == 1))
		// if legitimate row...
		if (ev.row < 0 || ev.row > dimsize (mafpc_patternspecs, 0) - 1)
			// don't do anything
		elseif (ev.col == 1)
			popupstr = "TRAIN;FINISH;PADFINISH"
			// allow amp & TTL based on control
			if (strsearch (mafpc_control, "DA", 0) != -1 || mafPC_isDC)
				popupstr = addlistitem ("AMP;RAMPSTART;RAMPEND", popupstr)
			endif
			if (findlistitem ("TTL", mafpc_control) != -1)
				popupstr = addlistitem ("TTL", popupstr)
			endif
			if (mafPC_isDC)
				popupstr = addlistitem ("DCPULSE", popupstr)
			endif
			popupstr = sortlist (popupstr)
			popupcontextualmenu popupstr	
			strswitch (s_selection)
				case "AMP":
				case "RAMPSTART":
				case "RAMPEND":
					mafPC_PatternSpecs[ev.row][1] = s_selection
					mafPC_PatternSel[ev.row][3] = 2
					if (!mafPC_isDC && (cmpstr ((mafPC_PatternSpecs[ev.row][2])[0], "G") != 0 || cmpstr ((mafPC_PatternSpecs[ev.row][2])[0,1], "DA") != 0))
						mafPC_PatternSpecs[ev.row][2] = "???"
					endif
					mafPC_PatternSpecs[ev.row][4] = ""
					if (mafPC_PatternSel[ev.row][4] == 0)
						mafPC_PatternSel[ev.row][4] = 0x20
					endif
					break
				case "TTL":
					mafPC_PatternSpecs[ev.row][1] = s_selection
					mafPC_PatternSpecs[ev.row][3] = ""
					mafPC_PatternSel[ev.row][3] = (mafPC_PatternSel[ev.row][3] | 0x20) & 0x30
					if (cmpstr ((mafPC_PatternSpecs[ev.row][2])[0,2], "TTL") != 0)
						mafPC_PatternSpecs[ev.row][2] = "???"
					endif
					mafPC_PatternSpecs[ev.row][4] = ""
					mafPC_PatternSel[ev.row][4] = 0
					break
				case "TRAIN":
					if (cmpstr (mafPC_PatternSpecs[ev.row][1], "TRAIN") == 0)		// already a train
						mafPC_param1 = stringfromlist (0, mafPC_PatternSpecs[ev.row][2])
						mafPC_param2 = stringfromlist (1, mafPC_PatternSpecs[ev.row][2])
					else
						mafPC_param1 = ""
						mafPC_param2="1"
					endif
					// bring up box
					NewPanel /W=(840.75,455.75,1037.25,542.75) as "Train"
					SetVariable setvar001,pos={11,31},size={153,19},proc=TrainRepsProc,title="# repetitions:"
					SetVariable setvar001,fSize=12,value= root:maf:mafPC_param2
					Button OKbutton,pos={87,61},size={50,20},proc=TrainOKCancelButtonProc,title="OK"
					Button Cancelbutton,pos={30,61},size={50,20},proc=TrainOKCancelButtonProc,title="Cancel"
					PopupMenu popup0,pos={0,6},size={150,24},title="Pattern name:",fSize=12
					PopupMenu popup0,mode=1,popvalue="",value= #"patternlist()"
					dowindow /c getTrainPatternPanel
					autopositionwindow /m=1/r=mafPC_EditPanel
					// set popup to previous value (if any)
					
					popupmenu Popup0 mode=(max (whichListItem (mafPC_param1, patternlist())+1, 1))
					pauseforuser getTrainPatternPanel
					if (mafPC_userResponse == 1)	// i.e. OK
						mafPC_PatternSpecs[ev.row][1] = s_selection
						mafPC_PatternSpecs[ev.row][2] = mafPC_param1 + ";" + mafPC_param2
						mafPC_PatternSpecs[ev.row][3,4] = ""
						mafPC_PatternSel[ev.row][3,4] = 0
					endif
					// open train window
					break
				case "DCPulse":
					mafPC_PatternSpecs[ev.row][1] = s_selection
					mafPC_PatternSel[ev.row][3] = 2
					if (cmpstr ((mafPC_PatternSpecs[ev.row][2])[0], "G") != 0)
						mafPC_PatternSpecs[ev.row][2] = "???"
					endif
					mafPC_PatternSpecs[ev.row][4] = ""
					mafPC_PatternSel[ev.row][4] = 0
					break
				case "FINISH":
				case "PADFINISH":
					mafPC_PatternSpecs[ev.row][1] = s_selection
					mafPC_PatternSpecs[ev.row][2,4] = ""
					mafPC_PatternSel[ev.row][2,4] = 0
					break
			endswitch
		elseif (ev.col == 2)
			popupstr = ""
			strswitch (mafpc_patternspecs[ev.row][1])
				case "AMP":
				case "RAMPSTART":
				case "RAMPEND":
					// set based on control
					popupstr = removefromlist ("TTL", mafPC_Control)
					if (mafPC_isDC)
						popupstr = addlistitem ("G1;G2;GForce", popupstr)
					endif
					break
				case "TTL":
					popupstr = "TTL0;TTL1;TTL2;TTL3"
					break
				case "DCPULSE":
					popupstr = "G1;G2"
					break
			endswitch
			if (strlen (popupstr) > 0)
				popupcontextualmenu popupstr
				if (v_flag > 0)
					mafPC_PatternSpecs[ev.row][ev.col] = stringfromlist (v_flag - 1, popupstr)
				endif
			endif
		endif
	endif
End
function EventButtonsProc(ctrlName) : ButtonControl
	String ctrlName
	wave mafPC_PatternSel=root:maf:mafPC_PatternSel
	wave /t mafPC_PatternSpecs=root:maf:mafPC_PatternSpecs, mafPC_PatternParams=root:maf:mafPC_PatternParams
	SVAR mafPC_param1=root:maf:mafPC_param1
	NVAR mafPC_UserResponse=root:maf:mafPC_UserResponse
	variable thisrow, i, swapval
	string swapstr

	strswitch (ctrlName)
		case "AddButton":
			thisrow = dimsize (mafPC_PatternSpecs, 0)
			InsertPoints /M=0 thisrow, 1, mafPC_PatternSel, mafPC_PatternSpecs, mafPC_PatternParams
			mafpc_patternspecs[thisrow][] = ""
			mafpc_patternsel[thisrow][0] = 2; mafpc_patternsel[thisrow][1,4] = 0
			mafpc_patternparams[thisrow] = ""
			Button Delbutton disable=0	// enable special buttons
			Button Sortbutton disable=0
			break
		case "DelButton":
			if (dimsize (mafPC_patternspecs, 0) == 1)	// no row to delete
				break
			endif
			thisrow = findSelectedRow (mafPC_PatternSel)	// illegal row selected
			if (thisrow == -1)
				break
			endif
			deletePoints /M=0 thisrow, 1, mafPC_PatternSel, mafPC_PatternSpecs, mafPC_PatternParams	
			if (dimsize (mafPC_patternspecs, 0) < 2)	// one row left, so disable special buttons
				Button Delbutton disable=2
				Button Sortbutton disable=2
			endif
			break
		case "SortButton":
			make /t/o/n=(dimsize (mafPC_PatternSpecs, 0)) mafPC_temp
			mafPC_temp = mafPC_PatternSpecs[p][0]
			sort2Dbetter ("root:maf:mafPC_temp", "root:maf:mafPC_PatternSpecs")
			sort2Dbetter ("root:maf:mafPC_temp", "root:maf:mafPC_PatternSel")
			sort /a mafPC_temp, mafPC_PatternParams
			break
		case "MoveUpButton":
			thisrow = findSelectedRow (mafPC_PatternSel)
			if (thisrow < 1 || thisrow > dimsize (mafPC_PatternSpecs, 0) - 1 || dimsize (mafPC_PatternSpecs, 0) == 1)
				break
			endif
			for (i = 0; i < dimsize (mafpc_patternspecs, 1); i += 1)
				swapstr = mafpc_PatternSpecs[thisrow][i]
				mafPC_PatternSpecs[thisrow][i] = mafPC_PatternSpecs[thisrow - 1][i]
				mafPC_PatternSpecs[thisrow - 1][i] = swapstr
				swapval = mafPC_PatternSel[thisrow][i]
				mafPC_PatternSel[thisrow][i] = mafPC_PatternSel[thisrow - 1][i]
				mafPC_PatternSel[thisrow - 1][i] = swapval
			endfor
			swapstr = mafPC_PatternParams[thisrow]
			mafPC_PatternParams[thisrow] = mafPC_PatternParams[thisrow - 1]
			mafPC_PatternParams[thisrow - 1] = swapstr
			listbox editlist selrow=thisrow - 1
			if (thisrow == 1)
				button MoveUpButton disable = 2
			endif
			button MoveDownButton disable = 0
			break
		case "MoveDownButton":
			thisrow = findSelectedRow (mafPC_PatternSel)
			if (thisrow < 0 || thisrow > dimsize (mafPC_PatternSpecs, 0) - 2 || dimsize (mafPC_PatternSpecs, 0) == 1)
				break
			endif
			for (i = 0; i < dimsize (mafpc_patternspecs, 1); i += 1)
				swapstr = mafpc_PatternSpecs[thisrow][i]
				mafPC_PatternSpecs[thisrow][i] = mafPC_PatternSpecs[thisrow + 1][i]
				mafPC_PatternSpecs[thisrow + 1][i] = swapstr
				swapval = mafPC_PatternSel[thisrow][i]
				mafPC_PatternSel[thisrow][i] = mafPC_PatternSel[thisrow + 1][i]
				mafPC_PatternSel[thisrow + 1][i] = swapval
			endfor
			swapstr = mafPC_PatternParams[thisrow]
			mafPC_PatternParams[thisrow] = mafPC_PatternParams[thisrow + 1]
			mafPC_PatternParams[thisrow + 1] = swapstr
			listbox editlist selrow=thisrow + 1
			if (thisrow == dimsize (mafPC_PatternSpecs, 0) - 2)
				button MoveDownButton disable = 2
			endif
			button MoveUpButton disable=0
			break
		case "WaveButton":
			// get current wave values, if any, and put into dialog field
			thisrow = findSelectedRow (mafPC_PatternSel)
			if (cmpstr ((mafPC_PatternSpecs[thisrow][3])[0,3], "WAVE") == 0)		// already a wave
				mafPC_param1 = stringbykey ("WAVE", mafPC_PatternSpecs[thisrow][3])
			else
				mafPC_param1 = ""
			endif
			// bring up box
			NewPanel /W=(693,503,889,590) as "Wave"
			SetVariable setvar0,pos={11,11},size={153,16},title="Wave name:"
			SetVariable setvar0,fSize=12,value= root:maf:mafPC_param1
			Button OKbutton,pos={87,61},size={50,20},proc=mafPC_OKCancelButtonProc,title="OK"
			Button Cancelbutton,pos={30,61},size={50,20},proc=mafPC_OKCancelButtonProc,title="Cancel"
			dowindow /c getWavePatternPanel
			autopositionwindow /m=1/r=mafPC_EditPanel
			controlupdate setvar0 
			SetVariable setvar0 activate
			pauseforuser getWavePatternPanel
			if (mafPC_userResponse == 0 || strlen (mafpc_param1) == 0)	// i.e. Cancel
				break
			endif
			mafPC_PatternSpecs[thisrow][3] = "WAVE:" + mafPC_param1
			break
	endswitch
end

//****************************************************
//	compiler routines
//****************************************************
function isNumber (teststr)
	string teststr
	variable firstdigit = char2num (teststr[0])
	return ((firstdigit >= char2num ("0") && firstdigit <= char2num ("9")) || firstdigit == char2num (".") || firstdigit == char2num ("+") || firstdigit == char2num ("-"))
end
function /t retrieveList (keyName, PatName, ParameterList)
	string keyName, PatName, ParameterList
	SVAR patStr=$("root:maf:" + PatName)
	string keyVal=replacestring (",", stringbykey (keyname, patStr), ";"), thiskey
	variable i

	for (i = 0; i < itemsinlist (keyVal); i += 1)
		thiskey = stringfromlist (i, keyval)
		if (isnumber (thiskey))	// if absolute number specified,
			// do nothing
		elseif (hasKey (thisKey, parameterList))	// variable specified in parameter list
			PatStr = replacestringbykey (thisKey, patstr, stringbykey (thisKey, parameterlist))
			keyval = replacestring (thiskey, keyval, stringbykey (thisKey, parameterlist), 0, 1)
		elseif (hasKey (thisKey, patstr))
			keyval = replacestring (thiskey, keyval, stringbykey (thisKey, patstr), 0, 1)
		else
			abort "Error compiling " + patName + ": Variable \"" + thiskey + "\" not specified"
		endif                         
	endfor
	return keyval
end
function hasKey (keyName, keyListStr)
	string keyName, keyListStr
	return (strlen (stringbykey (keyName, keyListStr)) > 0)
end
function getVariable (varName, PatName, ParameterList)
	string varName, PatName, ParameterList
	SVAR patStr=$("root:maf:" + PatName)
	
	if (hasKey (varName, parameterList))
		PatStr = replacestringbykey (varname, patstr, stringbykey (varName, parameterlist))
		return (numberbykey (varName, parameterList))
	elseif (hasKey (varName, patstr))
		return (numberbykey (varName, patstr))
	else
		abort "Error compiling " + patName + ": Variable \"" + varname + "\" not specified"
	endif                         
end
function retrieveValue (keyName, PatName, ParameterList)
	string keyName, PatName, ParameterList
	SVAR patStr=$("root:maf:" + PatName)
	if (strlen (stringbykey (keyname, patstr)) == 0)
		abort "Error compiling " + patName + ": No value specified for " + keyname
	elseif (isnumber (stringbykey (keyname, patstr)))	// if absolute number specified, return it
		return numberbykey (keyname, patstr)
	else	// variable specified, so get it
		return getvariable (stringbykey (keyname, patstr), patname, parameterlist)
	endif
end
function mafPC_CompilePattern (PatternName, ParameterList)
	// returns 1 if recompile needed; 0 otherwise
	string PatternName, ParameterList	
	SVAR mafPC_control=root:maf:mafPC_control
	variable i, needrecompile, tempval, tempval2, pstart, pend, j, k, finishTime, temptime, thishp, doPad
	string tempstr, thisvar, istr, chstr
	
	ParameterList = replacestring (" ", ParameterList, "")
	if (exists ("root:maf:" + PatternName) != 2)
		abort "Error compiling " + PatternName + ": Pattern does not exist"
	endif
	SVAR currPat = $("root:maf:" + PatternName)
	needrecompile = 1 - numberbykey ("COMPILED", currPat)	// does pattern think it is up to date?
	if (numtype (needrecompile) == 2)
		needrecompile = 1
	endif
	
	// go through dependencies
	tempstr = stringbykey ("DEPENDENCY", currPat)
	for (i = 0; i < itemsinlist (tempstr, ","); i += 1)
		if (mafPC_CompilePattern (stringfromlist (i, tempstr, ","), ParameterList) == 1)
			needrecompile = 1
		endif
	endfor
	if (!needrecompile)	// check whether included waves have changed since last compile
		tempstr = stringbykey ("WAVEDEPENDENCY", currPat)
		for (i = 0; i < itemsinlist (tempstr, ","); i += 1)
			if (moddate ($(stringfromlist (i, tempstr, ","))) > numberbykey ("COMPILETIME", currPat))
				needrecompile = 1
				break
			endif
		endfor
	endif
		
	if (!needrecompile)	// no need detected for dependencies, but check this one
		// check if any parameter in list is used by pattern, and if the value for that parameter has changed
		for (i = 0; i < itemsinlist (parameterlist); i += 1)
			thisvar = stringfromlist (0, stringfromlist (i, parameterlist), ":")
			if (strlen (stringbykey (thisvar,parameterlist)) == 0)
				abort "Error compiling " + patternName + ": Illegal parameter (" + thisvar + ") in pattern " + patternName
			endif
			if (cmpstr (stringbykey (thisvar, currpat), stringbykey (thisvar, parameterlist)) != 0)
				needrecompile = 1	// found bad match, so need recompile
				break
			endif
		endfor
	endif
	if (!needrecompile)
		return 0
	endif

	currPat = replacenumberbykey ("COMPILED", currPat, 0)	
	mafPC_control = replacestring (",", stringbykey ("CONTROL", currPat), ";")
	if (cmpstr (stringbykey ("PATTYPE", currPat), "Interval") == 0)
		// initialize destination waves
		for (k = 0; k < itemsinlist (mafPC_control); k += 1)
			make /n=0/o $("root:maf:" + patternname + "_" + stringfromlist (k, mafPC_control))
			setscale /P x, 0, numberbykey ("SAMPINT", currPat)/1e6, "s", $("root:maf:" + patternname + "_" + stringfromlist (k, mafPC_control))
		endfor
		for (i = 0; i < numberbykey ("NUMROWS", currPat); i += 1)
			istr = num2str (i)
			if (cmpstr (stringbykey ("TIME" + istr, currPat), "TRAIN") == 0)
				tempval = retrieveValue ("REP" + istr, PatternName, ParameterList)
				if (numtype (tempval) != 0 || tempval != round (tempval) || tempval < 0)
					abort "Error compiling " + patternName + ": Illegal repetition number (REP" + istr + " = " + num2str (tempval) + ")"
				endif
				tempstr = stringbykey ("PATTERN" + istr, currPat)
				for (k = 0; k < itemsinlist (mafPC_control); k += 1)
					if (exists ("root:maf:" + tempstr + "_" + stringfromlist (k, mafPC_control)) != 1)
						abort "Error compiling " + patternName + ": Could not locate train component \"" +  tempstr + "_" + stringfromlist (k, mafPC_control) + "\""
					endif					
					wave whp = $("root:maf:" + patternname + "_" + stringfromlist (k, mafPC_control))
					pstart = numpnts (whp)
					wave wthp = $("root:maf:" + tempstr + "_" + stringfromlist (k, mafPC_control))
					if (deltax (wthp)*1e6 != deltax (whp)*1e6)
						abort "Error compiling " + patternName + ": Sampling rates do not match (" + patternname + " = " + num2str (deltax (whp) * 1e6) + " vs. " + tempstr + " = " + num2str (deltax (wthp)) + ")"
					endif
					redimension /n=(pstart + numpnts (wthp) * tempval ) whp
					for (j = 0; j < tempval; j += 1)	// could also be clever and use mod function
						pend = pstart + numpnts (wthp) - 1
						whp[pstart, pend] = wthp[p - pstart]
						pstart = pend + 1
					endfor
				endfor
			else
				temptime = retrieveValue ("TIME" + istr, PatternName, ParameterList)
				if (numtype (temptime) != 0 || temptime < 0)
					abort "Error compiling " + patternName + ": Illegal time duration (TIME" + istr + " = " + num2str (temptime) + ")"
				endif
				for (k = 0; k < itemsinlist (mafPC_control); k += 1)
					wave whp=$("root:maf:" + patternname + "_" + stringfromlist (k, mafPC_control))
					pstart = numpnts (whp)
					redimension /n=(pstart + temptime * 1e-3 / deltax(whp)) whp
					if (cmpstr (stringfromlist (k, mafPC_control), "TTL") == 0)
						tempval = numberbykey ("TTL" + istr, currPat)
						whp[pstart, *] = tempval
					else
						chstr = thischannum (k, mafPC_control)
						thishp = (numberbykey ("ABS" + istr + "_" + chstr, currPat) == 0) ? getVariable ("HP" + chstr, PatternName, ParameterList) : 0
						if (cmpstr (stringbykey ("AMP" + istr + "_" + chstr, currPat), "RAMP") == 0)
							tempval = retrieveValue ("RAMPSTART" + istr + "_" + chstr, PatternName, ParameterList) + thisHP
							tempval2 = retrieveValue ("RAMPEND" + istr + "_" + chstr, PatternName, ParameterList) + thisHP
							tempval2 = (tempval2 - tempval) / (numpnts (whp) - 1 - pstart)	// recast parameters as start and slope
							whp[pstart, *] = tempval2 * (p - pstart) + tempval
						elseif (cmpstr (stringbykey ("AMP" + istr + "_" + chstr, currPat), "WAVE") == 0)
							tempstr = stringbykey ("WAVE" + istr + "_" + chstr, currPat)
							if (exists (tempstr) != 1)
								abort "Error compiling "+ patternName + ": Couldn't locate amplitude wave \"" + tempstr + "\""
							endif
							wave wthp=$tempstr
							whp[pstart, *] = wthp[p - pstart] + thishp
							if (numpnts (wthp) < numpnts (whp) - pstart)
								print "Warning compiling " + patternname + ": Wave \"" + tempstr + "\" does not fill time allotted.  Padding with last value."
							elseif (numpnts (wthp) > numpnts (whp) - pstart)
								print "Warning compiling " + patternname + ": Wave \"" + tempstr + "\" too long for time allotted.  Truncated."
							endif
						else
							tempval = retrieveValue ("AMP" + istr + "_" + chstr, PatternName, ParameterList) + thishp
							whp[pstart, *] = tempval
						endif
					endif
				endfor
			endif
		endfor
	elseif (cmpstr (stringbykey ("PATTYPE", currPat), "Event") == 0)
		// do pre-compile
		make /t/n=0/o root:maf:mafPC_tempCompileEvent, root:maf:mafPC_tempCompileCh, root:maf:mafPC_tempCompileAmp
		make /n=0/o root:maf:mafPC_tempCompileTime, root:maf:mafPC_tempCompileAbs
		wave/t wtempEvent = root:maf:mafPC_tempCompileEvent, wtempCh=root:maf:mafPC_tempCompileCh, wtempAmp=root:maf:mafPC_tempCompileAmp
		wave wtempTime = root:maf:mafPC_tempCompileTime, wtempAbs=root:maf:mafPC_tempCompileAbs
		finishtime = 0; dopad = 0; temptime = 0
		for (i = 0; i < numberbykey ("NUMROWS", currPat); i += 1)
			istr = num2str (i)
			tempstr = replacestring (",", retrieveList ("TIME" + istr, PatternName, ParameterList), ";")
			tempval = itemsinlist (tempstr)
			if (tempval == 0)
				abort "Error compiling " + PatternName + ": No event time specified (TIME" + istr + ")"
			endif
			insertpoints 0, tempval, wtempTime, wtempEvent, wtempCh, wtempAmp, wtempAbs
			wtempTime[0, tempval - 1] = str2num (stringfromlist (p, tempstr))
			wavestats /q/r=[0, tempval - 1] wtemptime	// stores the last time into v_max
			wtempEvent[0, tempval - 1] = stringbykey ("EVENT" + istr, currPat)
			strswitch (wtempEvent[0])
				case "TRAIN":
					wtempCh[0, tempval - 1] = stringbykey ("PATTERN" + istr, currPat)
					wtempAmp[0, tempval - 1] = num2str (retrieveValue ("REP" + istr, PatternName, ParameterList))
					SVAR subpat = $("root:maf:" + wtempCh[0])
					wave w=$("root:maf:" + wtempCh[0] + "_" + stringfromlist (0, stringbykey ("CONTROL", subpat), ","))
					temptime = max (temptime, v_max + 1e3 * deltax (w) * numpnts (w) * str2num (wtempAmp[0]))
					break
				case "AMP":
				case "RAMPSTART":
				case "RAMPEND":
					wtempAbs[0, tempval - 1] = numberbykey ("ABS" + istr, currPat)
				case "TTL":
				case "DCPULSE":
					wtempCh[0, tempval - 1] = stringbykey ("CH" + istr, currPat)
					if (cmpstr ("WAVE", stringbykey ("AMP" + istr, currPat)) == 0)
						wtempAmp[0, tempval - 1] = "WAVE:" + stringbykey ("WAVE" + istr, currPat)
						wave w=$(wtempAmp[0])
						temptime = max (temptime, v_max + 1e3 * deltax (w) * numpnts (w))
					else
						temptime = max (temptime, v_max)
						tempstr = replacestring (",", retrieveList ("AMP" + istr, PatternName, ParameterList), ";")
						if (itemsinlist (tempstr) == 0)
							abort "Error compiling " + PatternName + ": No amplitude specified (AMP" + istr + ")"
						elseif (itemsinlist (tempstr) == 1)
							wtempAmp[0, tempval - 1] = tempstr
						elseif (itemsinlist (tempstr) == tempval)
							wtempAmp[0, tempval - 1] = stringfromlist (p, tempstr)
						else
							abort "Error compiling " + PatternName + ": Mismatch between number of times (" + num2str (tempval) + ") and amplitudes (" + num2str (itemsinlist (tempstr)) + ") in step " + istr
						endif
					endif
					break
				case "FINISH":
					if (finishTime != 0)
						abort "Error compiling " + PatternName + ": Multiple FINISH events"
					endif
					finishTime = wtempTime[0]
					deletepoints /M=0 0, tempval, wtempTime, wtempEvent, wtempCh, wtempAmp, wtempAbs	// remove from overall list
					break
				case "PADFINISH":
					if (finishTime != 0)
						abort "Error compiling " + PatternName + ": Multiple FINISH events"
					endif
					dopad = 1
					finishTime = wtemptime[0]
					deletepoints /M=0 0, tempval, wtempTime, wtempEvent, wtempCh, wtempAmp, wtempAbs	// remove from overall list
					break
			endswitch
		endfor
		sort wtempTime, wtempTime, wtempEvent, wtempCh, wtempAmp, wtempAbs
		if (doPad)	// i.e. using PADFINISH
			finishTime += temptime
		endif
		if (finishTime == 0)
			abort "Error compiling " + PatternName + ": No FINISH event found"
		endif
		// initialize destination waves
		for (i = 0; i < itemsinlist (mafPC_control); i += 1)
			make /o/n=(round (finishTime * 1e3 / numberbykey ("SAMPINT", currPat) + 1)) $("root:maf:" + patternname + "_" + stringfromlist (i, mafPC_control))
			wave whp=$("root:maf:" + patternname + "_" + stringfromlist (i, mafPC_control))
			whp = 0
			setscale /P x, 0, numberbykey ("SAMPINT", currPat)/1e6, "s", whp
		endfor
		// now go through sorted and expanded event list and pile into destination waves
		for (i = 0; i < numpnts (wtempevent); i += 1)
			if (wtempTime[i] > finishTime)
				print "Warning: Some events in " + PatternName + " occur after FINISH time."
				break
			endif
			strswitch (wtempEvent[i])
				case "TRAIN":
					SVAR subPat = $("root:maf:" + wtempCh[i])
					tempstr = replacestring (",", stringbykey ("CONTROL", subpat), ";")
					for (j = 0; j < itemsinlist (tempstr); j += 1)
						if (exists ("root:maf:" + wtempCh[i] + "_" + stringfromlist (j, tempstr)) != 1)
							abort "Error compiling " + patternName + ": Can't locate subpattern component \"" + wtempCh[i] + "_" + stringfromlist (j, tempstr) + "\""
						endif
						if (exists ("root:maf:" + patternname + "_" + stringfromlist (j, tempstr)) != 1)
							abort "Error compiling " + patternName + ": Subpattern \"" + wtempCh[i] + "\" controls " + stringfromlist (j, tempstr) + ", but \"" + patternname + "\" does not."
						endif
						wave whp = $("root:maf:" + patternname + "_" + stringfromlist (j, tempstr))
						wave wthp = $("root:maf:" + wtempCh[i] + "_" + stringfromlist (j, tempstr))
						if (deltax (wthp) * 1e6 != deltax (whp) * 1e6)
							abort "Error compiling " + patternName + ": Sampling rate of train \"" + tempstr + "\" does not match (" + num2str (deltax (whp) * 1e6) + " vs. " + num2str (deltax (wthp)) + ")"
						endif
						pstart = x2pnt (whp, wtempTime[i]/1000)
						for (k = 0; k < str2num (wtempAmp[i]); k += 1)
							pend = pstart + numpnts (wthp) - 1
							if (cmpstr (stringfromlist (j, tempstr), "TTL") == 0)
								whp[pstart, pend] = whp[p] | wthp[p - pstart]
							else
								whp[pstart, pend] = wthp[p - pstart]
							endif
							pstart = pend + 1
						endfor
					endfor
					break
				case "AMP":
					wave whp = $("root:maf:" + patternname + "_" + wtempCh[i])
					pstart = x2pnt (whp, wtempTime[i]/1000)
					thishp = (wtempAbs[i] || cmpstr ((wtempCh[i])[0,1], "DA") != 0) ? 0 : getVariable ("HP" + (wtempCh[i])[2], PatternName, ParameterList)
					if (haskey ("WAVE", wtempAmp[i]))
						tempstr = stringbykey ("WAVE", wtempAmp[i])
						if (exists (tempstr) != 1)
							abort "Error compiling " + patternName + ": Couldn't locate amplitude wave \"" + tempstr + "\""
						endif
						wave wthp=$tempstr
						whp[pstart, *] = wthp[p - pstart] + thishp
						if (pstart + numpnts (wthp) - 1 > numpnts (whp) - 1)
							print "Warning compiling " + patternname + ": Wave \"" + tempstr + "\" too long for time allotted.  Truncated."
						endif
					else
						tempval = str2num (wtempAmp[i]) + thishp
						whp[pstart, *] = tempval
					endif
					break					
				case "TTL":
					wave whp = $("root:maf:" + patternname + "_TTL")
					pstart = x2pnt (whp, wtempTime[i]/1000)
					tempval = 2^(str2num ((wtempCh[i])[3]))
					if (str2num (wtempAmp[i]))
						whp[pstart, inf] = whp[p] | tempval
					else
						whp[pstart, inf] = whp[p] & ~tempval
					endif
					break
				case "DCPULSE":
					wave whp = $("root:maf:" + patternname + "_" + wtempCh[i])
					tempstr = stringbykey ("DCMINI" + (wtempCh[i])[1], currPat)
					if (exists (tempstr) != 1)
						abort "Error compiling " + patternName + ": Couldn't locate mini wave \"" + tempstr + "\""
					endif
					wave wthp=$tempstr
					if (deltax (wthp) * 1e6 != deltax (whp) * 1e6)
						abort "Error compiling " + patternName + ": Sampling rate of dynamic clamp mini \"" + tempstr + "\" does not match (" + num2str (deltax (whp) * 1e6) + " vs. " + tempstr + " = " + num2str (deltax (wthp) * 1e6) + ")"
					endif
					pstart = x2pnt (whp, wtempTime[i]/1000)
					tempval = str2num (wtempAmp[i])
					whp[pstart, *] += wthp[p - pstart] * tempval
					break
				case "RAMPSTART":
					// find first matching RAMPEND statement
					for (j = i + 1; j < numpnts (wtempevent); j += 1)
						if (cmpstr (wtempEvent[j], "RAMPEND") == 0 && cmpstr (wtempCh[i], wtempCh[j]) == 0)
							break
						endif
					endfor
					if (j == numpnts (wtempevent))	// i.e. ran off the end
						abort "Error compiling " + patternname + ": RAMPSTART with no matching RAMPEND"
					endif
					wave whp = $("root:maf:" + patternname + "_" + wtempCh[i])
					tempval = str2num (wtempAmp[i]) + ((wtempAbs[i] || cmpstr ((wtempCh[i])[0,1], "DA") != 0) ? 0 : getVariable ("HP" + (wtempCh[i])[2], PatternName, ParameterList))
					tempval2 = str2num (wtempAmp[j]) + ((wtempAbs[j] || cmpstr ((wtempCh[i])[0,1], "DA") != 0) ? 0 : getVariable ("HP" + (wtempCh[i])[2], PatternName, ParameterList))
					pstart = x2pnt (whp, wtempTime[i]/1000)
					pend = x2pnt (whp, wtempTime[j]/1000)
					tempval2 = (tempval2 - tempval) / (pend - pstart)	// recast start and end values into slope
					whp[pstart, pend] = tempval2 * (p - pstart) + tempval
					whp[pend + 1, *] = whp[pend]
					wtempEvent[j] = "wasRAMPEND"
					break
				case "RAMPEND":
					abort "Error compiling " + patternname + ": RAMPEND with no matching RAMPSTART"
					break
			endswitch
		endfor
	endif
	currPat = replacenumberbykey ("COMPILED", currPat, 1)
	currPat = replacenumberbykey ("COMPILETIME", currPat, datetime)
	// make showTTL waves for verify	-- too slow.  need to make fast
	if (exists ("root:maf:" + patternname + "_TTL"))
		wave w=$("root:maf:" + patternname + "_TTL")
		make /o/n=0 $("root:maf:" + patternname + "_showTTLx"), $("root:maf:" + patternname + "_showTTLy")
		wave wx=$("root:maf:" + patternname + "_showTTLx"), wy=$("root:maf:" + patternname + "_showTTLy")
		make /o/n=7 root:maf:tempshow
		wave tempshow = root:maf:tempshow
		tempshow = -1	// holds last time TTL turned on
		tempval = 0
		for (i = 0; i < numpnts (w); i += 1)
			if (tempval != w[i])	// something has changed, so update times
				for (j = 0; j < 7; j += 1)
					if ((tempval & 2^j) != (w[i] & 2^j))	// this bit has changed
						if (tempshow[j] == -1)	// i.e. just starting
							tempshow[j] = i
						else	// just finishing, so add to plot
							redimension /n=(numpnts (wx) + 3) wx, wy
							wx[numpnts (wx) - 3] = tempshow[j]; wx[numpnts (wx) - 2] = i - 1; wx[numpnts (wx) - 1] = nan
							wy[numpnts (wx) - 3, numpnts (wx) - 2] = j; wy[numpnts (wx) - 1] = nan
							tempshow[j] = -1	// mark as completed
						endif
					endif
				endfor
				tempval = w[i]
			endif
		endfor
		for (j = 0; j < 7; j += 1)		// wrap up TTLs that didn't come back down at the end
			if (tempshow[j] != -1)
				redimension /n=(numpnts (wx) + 3) wx, wy
				wx[numpnts (wx) - 3] = tempshow[j]; wx[numpnts (wx) - 2] = i; wx[numpnts (wx) - 1] = nan
				wy[numpnts (wx) - 3, numpnts (wx) - 1] = j
			endif
		endfor
		redimension /n=(numpnts (wx) + 1) wx, wy	// add one last point to make sure x-scaling goes out to end
		wx[numpnts (wx) - 1] = i
		wy[numpnts (wx) - 1] = nan
		wx = numtype (wx) == 0 ? pnt2x (w, wx) : nan
	endif
	return 1
end

function mafPC_VerifyCompile (patname)
	string patname
	svar currPat=$("root:maf:" + patname)
	string control = replacestring (",", stringbykey ("CONTROL", currPat), ";")
	variable i, numch = itemsinlist (control)
	string an
	
	if (numberbykey ("COMPILED", currPat) != 1)
		abort "Pattern must be compiled before verifying."
	endif
	display
	for (i = 0; i < numch; i += 1)
		an="l" + num2str (i)
		if (cmpstr (stringfromlist (i, control), "TTL") == 0)
			appendtograph /l=$an $("root:maf:" + patname + "_showTTLy") vs $("root:maf:" + patname + "_showTTLx")
			modifygraph mode[i]=0, lsize[i]=3
		else
			appendtograph /l=$an $("root:maf:" + patname + "_" + stringfromlist (i, control))
			modifygraph mode[i]=6
		endif
		modifygraph axisenab($an)={1.05/numch * i, 1.05 * (i + 1) / numch - .05}
		label $an stringfromlist (i, control)
		ModifyGraph lblPos($an)=40,freePos($an)=0
	endfor
	modifygraph margin(left)=40
	textbox /f=0/a=mt/e/b=1 patname
end

function /t retrieveAllParameters (patname)
	string patname
	svar currpat=$("root:maf:" + patname)
	string allparams="", istr, varname, subparams, control, liststr, ch
	variable i, j, numch, daval, isInterval
	
	control = sortlist (replacestring (",", stringbykey ("CONTROL", currPat), ";"))
	numch = countDA (control)
	if (cmpstr (stringbykey ("PATTYPE", currPat), "Interval") == 0)
		for (i = 0; i < numberbykey ("NUMROWS", currpat); i += 1)
			istr = num2str(i)
			if (cmpstr (stringbykey ("TIME" + istr, currpat), "TRAIN") == 0)
				// add parameters from train pattern
				subparams = retrieveAllParameters (stringbykey ("PATTERN" + istr, currPat))
				for (j = 0; j < itemsinlist (subparams); j += 1)
					allparams = replacestringbykey (stringfromlist (0, stringfromlist (j, subparams), ":"), allparams, stringfromlist (1, stringfromlist (j, subparams), ":"))
				endfor
				if (!isnumber (stringbykey ("REP" + istr, currPat)))
					varname = stringbykey ("REP" + istr, currPat)
					allparams = replacestringbykey (varname, allparams, stringbykey (varname, currpat))
				endif
			else
				if (!isnumber (stringbykey ("TIME" + istr, currPat)))
					varname = stringbykey ("TIME" + istr, currPat)
					allparams = replacestringbykey (varname, allparams, stringbykey (varname, currpat))
				endif
				for (j = 0; j < numch; j += 1)
					ch = thischannum (j, control)
					if (cmpstr (stringbykey ("AMP" + istr + "_" + ch, currpat), "RAMP") == 0)
						if (!isnumber (stringbykey ("RAMPSTART" + istr + "_" + ch, currPat)))
							varname = stringbykey ("RAMPSTART" + istr + "_" + ch, currpat)
							allparams = replacestringbykey (varname, allparams, stringbykey (varname, currpat))
						endif
						if (!isnumber (stringbykey ("RAMPEND" + istr + "_" + ch, currPat)))
							varname = stringbykey ("RAMPEND" + istr + "_" + ch, currpat)
							allparams = replacestringbykey (varname, allparams, stringbykey (varname, currpat))
						endif
					elseif (cmpstr ("WAVE", stringbykey ("AMP" + istr + "_" + ch, currPat)) == 0)
						// do nothing
					elseif (!isnumber (stringbykey ("AMP" + istr + "_" + ch, currpat)))
						varname = stringbykey ("AMP" + istr + "_" + ch, currpat)
						allparams = replacestringbykey (varname, allparams, stringbykey (varname, currpat))
					endif
				endfor
			endif
		endfor
	elseif (cmpstr (stringbykey ("PATTYPE", currPat), "Event") == 0)
		for (i = 0; i < numberbykey ("NUMROWS", currpat); i += 1)
			istr = num2str(i)
			liststr = replacestring (",", stringbykey ("TIME" + istr, currPat), ";")
			for (j = 0; j < itemsinlist (liststr); j += 1)
				varname = stringfromlist (j, liststr)
				if (!isnumber (varname))
					allparams = replacestringbykey (varname, allparams, stringbykey (varname, currPat))
				endif
			endfor
			strswitch (stringbykey ("EVENT" + istr, currPat))
				case "TRAIN":
					subparams = retrieveAllParameters (stringbykey ("PATTERN" + istr, currPat))
					for (j = 0; j < itemsinlist (subparams); j += 1)
						allparams = replacestringbykey (stringfromlist (0, stringfromlist (j, subparams), ":"), allparams, stringfromlist (1, stringfromlist (j, subparams), ":"))
					endfor
					if (!isnumber (stringbykey ("REP" + istr, currPat)))
						varname = stringbykey ("REP" + istr, currPat)
						allparams = replacestringbykey (varname, allparams, stringbykey (varname, currpat))
					endif
					break
				case "DCPULSE":
				case "AMP":
				case "RAMPSTART":
				case "RAMPEND":
					liststr = replacestring (",", stringbykey ("AMP" + istr, currPat), ";")
					if (cmpstr ("WAVE", liststr) == 0)
						break
					endif
					for (j = 0; j < itemsinlist (liststr); j += 1)
						varname = stringfromlist (j, liststr)
						if (!isnumber (varname))
							allparams = replacestringbykey (varname, allparams, stringbykey (varname, currPat))
						endif
					endfor
					break
			endswitch
		endfor
	endif
	return allparams
end
function /t mafPC_EnterParameters (patname)
	string patname
	svar currPat=$("root:maf:" + patname)
	string allparams = retrieveallparameters (patname)
	variable i, numvars
	NVAR mafPC_userResponse=root:maf:mafPC_userResponse
	
	if (modalConflict ("", ""))
		return ""
	endif
	if (itemsinlist (allparams) == 0)	// no items to set
		mafPC_userResponse = 1
		return ""
	endif
	// set up waves to retrieve data
	allparams = sortlist (allparams, ";", 4)
	numvars = itemsinlist (allparams)
	make /t/o/n=(numvars, 2) root:maf:mafPC_ParamVals
	make /o/n=(numvars, 2) root:maf:mafPC_ParamSel
	wave /t mafPC_ParamVals=root:maf:mafPC_ParamVals; wave mafPC_ParamSel=root:maf:mafPC_ParamSel
	mafPC_ParamVals[0, numvars - 1][] = stringfromlist (q, stringfromlist (p, allparams), ":")
	mafPC_ParamSel[0, numvars - 1][0] = 0
	mafPC_ParamSel[0, numvars - 1][1] = 2
	setdimlabel 1, 0, Variable, mafpc_paramvals
	setdimlabel 1, 1, Value, mafpc_paramvals
	
	NewPanel /W=(150,50,341.25,251)
	dowindow /c/t mafPC_ParamPanel, "Enter Parameters"
	SetDrawLayer UserBack
	ListBox ParamList,pos={9,9},size={175,162},proc=mafPC_varlistboxproc,frame=3
	ListBox ParamList,listWave=root:maf:mafPC_ParamVals,selWave=root:maf:mafPC_ParamSel
	ListBox ParamList,mode= 5
	Button Runbutton,pos={40,176},size={50,20},title="Run", proc=mafPC_OKCancelButtonProc
	Button Cancelbutton,pos={93,176},size={50,20},title="Cancel", proc=mafPC_OKCancelButtonProc
	pauseforuser mafPC_ParamPanel
	if (mafPC_userResponse == 0)
		return ""
	else
		for (i = 0; i < numvars; i += 1)
			allparams = replacestringbykey (mafPC_ParamVals[i][0], allparams, mafPC_ParamVals[i][1])
		endfor
		return (allparams)
	endif
end
Function mafPC_VarListboxProc(ctrlName,row,col,event)
	String ctrlName     // name of this control
	Variable row,col,event
	variable onebad=0, i, firstdigit
	wave /t mafPC_ParamVals=root:maf:mafPC_ParamVals
	//check to see if all values are okay
	for (i = 0; i < dimsize (mafPC_ParamVals, 0); i += 1)
		firstdigit = char2num ((mafpc_paramvals[i][1])[0])
		if ((numtype (firstdigit) != 0 || firstdigit < char2num ("0") || firstdigit > char2num ("9")) && firstdigit != char2num (".") && firstdigit != char2num ("+") && firstdigit != char2num ("-"))
			onebad = 1
			break
		endif
	endfor
	button RunButton disable=onebad * 2
end


////////////////////////////////////////////////
//	I/O functions
////////////////////////////////////////////////
function mafPC_SavePattern (patname)
	string patname	// may also be list of patterns, e.g. "pat1;pat2;pat3"
	variable savefile
	variable i
	string thispat
	
	for (i = 0; i < itemsinlist (patname); i += 1)
		thispat = stringfromlist (i, patname)
		open /p=mafPCPath/T="TEXT" savefile as thispat +".pat"
//		NVAR v_flag
		if (savefile == 0)
			abort "Error saving " + thispat	// maybe don't abort?***
		endif
		SVAR currPat = $("root:maf:" + thispat)
		fbinwrite savefile, currPat	// error checking?
		close savefile
	endfor
end
function LoadPatternListProc (ctrlName, row, col, event)
	string ctrlName
	variable row, col, event
	button LoadButton disable=((findselectedrow (root:maf:mafPC_LoaderSel) != -1) ? 0 : 2)
end
function mafPC_LoadPattern (patname)
	string patname	// may be string list, e.g. "pat1;pat2;pat3"
	wave /t mafPC_Patterns=root:maf:mafPC_Patterns
	NVAR mafPC_userResponse=root:maf:mafPC_userResponse
	variable openfile, i, needload
	string tempstr, thispat
	
	if (modalConflict ("", ""))
		return 0
	endif
	if (strlen (patname) == 0)	// was no pattern specified?
		tempstr = indexedfile (mafPCPath, -1, ".pat")
		if (itemsinlist (tempstr) == 0)
			doalert 0, "No patterns available to load"
			return 0
		endif
		make /t/o/n=(itemsinlist (tempstr)) root:maf:mafPC_LoaderWave; wave /t mafPC_LoaderWave=root:maf:mafPC_LoaderWave
		make /o/n=(itemsinlist (tempstr)) root:maf:mafPC_LoaderSel; wave mafPC_LoaderSel=root:maf:mafPC_LoaderSel
		mafPC_LoaderWave = filenameonly (stringfromlist (p, tempstr))
		mafPC_LoaderSel = 0
		sort mafPC_LoaderWave, mafPC_LoaderWave
		NewPanel /W=(579,407.75,750.75,557.75) as "Load Pattern"
		dowindow /c LoadPattern
		SetDrawLayer UserBack
		DrawText 0,19,"Select the pattern(s) to load:"
		ListBox loadlist,pos={15,23},size={150,90}
		ListBox loadlist,listWave=root:maf:mafPC_LoaderWave,selWave=root:maf:mafPC_LoaderSel
		ListBox loadlist,mode= 4,editStyle= 1, proc=loadPatternListProc
		Button Loadbutton,pos={30,125},size={50,20},disable=2,title="Load",proc=mafPC_OKCancelButtonProc
		Button Cancelbutton,pos={93,125},size={50,20},title="Cancel", proc=mafPC_OKCancelButtonProc
		autopositionwindow /m=1/r=mafPC_PatternPanel
		pauseforuser LoadPattern
		if (mafPC_userResponse == 0)
			return 0
		endif
		for (i = 0; i < numpnts (mafPC_LoaderWave); i += 1)
			if (mafPC_LoaderSel[i])
				patname += mafPC_LoaderWave[i] + ";"
			endif
		endfor
	endif
	
	// go through pattern list and load them
	for (i = 0; i < itemsinlist (patname); i += 1)
		// check if opened file already exists
		thispat = stringfromlist (i, patname)
		needload = 1
		if (exists ("root:maf:" + thispat) == 2)
			doalert 1, "Pattern \"" + thispat + "\" already loaded.  Do you wish to overwrite it?"
			if (v_flag == 2)	// i.e. no
				needload = 0
			endif
		else	
			string /g $("root:maf:" + thispat)
		endif
		if (findwavestrval (mafPC_Patterns, thispat) == -1)	// do we need to insert it into pattern list?
			insertpoints numpnts (mafPC_Patterns), 1, mafPC_Patterns
			mafPC_Patterns[numpnts (mafPC_Patterns) - 1] = thispat	
		endif
		if (needload)
			open /z/p=mafPCPath/r/t="TEXT" openfile as thispat + ".pat"
			if (openfile == 0)
				abort "Unable to load pattern \"" + thispat + "\"."
			endif
			svar currPat = $("root:maf:" + thispat)
			freadline openfile, currPat
			close openfile
			currpat = replacenumberbykey ("COMPILED", currPat, 0)
		endif
	endfor
end

function mafPC_NewSet ()
	mafPC_Init()
	wave /t mafPC_Patterns=root:maf:mafPC_Patterns
	SVAR mafPC_SetName=root:maf:mafPC_SetName

	if (numpnts (mafPC_Patterns) > 0)
		doalert 1, "Clear existing set?"
		if (v_flag == 2)
			return 0
		endif
	endif
	redimension /n=0 mafPC_Patterns
	dowindow /f mafPC_PatternPanel
	if (v_flag == 0)
		execute "mafPC_PatternPanel()"
	endif
	mafPC_SetName = ""
	dowindow /t mafPC_PatternPanel, "Untitled Set"
end
Function OpenSetButtonProc(ctrlName) : ButtonControl
	String ctrlName
	NVAR mafPC_userResponse=root:maf:mafPC_userResponse
	SVAR mafPC_SetName=root:maf:mafPC_SetName
	wave /t mafPC_LoaderWave=root:maf:mafPC_LoaderWave
	
	mafPC_userResponse = (cmpstr (ctrlName, "OpenButton") == 0) ? 1 : 0
	if (mafPC_userResponse)
		mafPC_SetName = mafPC_LoaderWave[checkval ("loadList","")]
	endif
	DoWindow /K OpenSet
End
function LoadSetListProc (ctrlName, row, col, event)
	string ctrlName
	variable row, col, event
	wave /t mafPC_Patterns=root:maf:mafPC_Patterns
	variable d=2
	d = (row >= 0 && row < numpnts (root:maf:mafPC_LoaderWave)) ? 0 : 2
	button OpenButton disable=d
	return 0
end

function /t mafPC_CurrentSet ()
	if (exists ("root:maf:mafPC_SetName") == 2)
		SVAR mafPC_SetName=root:maf:mafPC_SetName
		return mafPC_SetName
	else
		return ""
	endif
end

function mafPC_OpenSet (setname)
	string setname
	variable openfile
	string patlist, tempstr
	
	mafPC_Init()
	wave /t mafPC_Patterns=root:maf:mafPC_Patterns
	SVAR mafPC_SetName=root:maf:mafPC_SetName
	NVAR mafPC_UserResponse=root:maf:mafPC_UserResponse
	if (modalConflict ("", ""))
		return 0
	endif
	if (numpnts (mafPC_Patterns) > 0)
		doalert 1, "Overwrite existing set?"
		if (v_flag == 2)
			return 0
		endif
	endif
	if (strlen (setname) == 0)	// no set specified, so go looking
		tempstr = indexedfile (mafPCPath, -1, ".set")
		if (itemsinlist (tempstr) == 0)
			doalert 0, "No sets available to open."
			return 0
		endif
		make /t/o/n=(itemsinlist (tempstr)) root:maf:mafPC_LoaderWave; wave /t mafPC_LoaderWave=root:maf:mafPC_LoaderWave
		mafPC_LoaderWave = filenameonly (stringfromlist (p, tempstr))
		sort mafPC_LoaderWave, mafPC_LoaderWave
		NewPanel /W=(579,407.75,750.75,557.75) as "Open Set"
		dowindow /c OpenSet
		SetDrawLayer UserBack
		DrawText 0,19,"Select the set to open:"
		ListBox loadlist,pos={15,23},size={150,90}
		ListBox loadlist,listWave=root:maf:mafPC_LoaderWave
		ListBox loadlist,mode= 2,editStyle= 1, proc=LoadSetListProc
		Button Openbutton,pos={30,125},size={50,20},disable=2,title="Open",proc= OpenSetButtonProc
		Button Cancelbutton,pos={93,125},size={50,20},title="Cancel", proc= OpenSetButtonProc
		dowindow mafPC_PatternPanel
		if (v_flag == 1)
			autopositionwindow /m=1/r=mafPC_PatternPanel
		endif
		pauseforuser OpenSet
		if (mafPC_userResponse == 0)
			return 0
		endif
		setname = mafPC_SetName 
	endif
	open /z/p=mafPCPath/R/T="TEXT" openfile as setname + ".set"
	if (openfile == 0)
		abort "Unable to open set \"" + setname + "\"."
	endif
	redimension /n=0 mafPC_Patterns
	freadline openfile, patlist
	close openfile
	if (strlen (patlist) > 0)
		mafPC_LoadPattern (patlist)
	endif
	mafPC_SetName = setname
	dowindow /f mafPC_PatternPanel
	if (v_flag == 0)
		execute "mafPC_PatternPanel ()"
	endif
	dowindow /t mafPC_PatternPanel, mafPC_SetName
end
Function SaveSetFieldProc (ctrlName,varNum,varStr,varName) : SetVariableControl
	String ctrlName, varStr, varName
	Variable varNum
	button savebutton disable= (strlen (varstr) > 0 ? 0 : 2)
end
function mafPC_SaveSet (setname)
	string setname
	wave /t mafPC_Patterns=root:maf:mafPC_Patterns
	SVAR mafPC_SetName=root:maf:mafPC_SetName
	NVAR mafPC_userResponse=root:maf:mafPC_userResponse
	variable savefile, i
	string patlist
	
	if (modalConflict ("", ""))
		return 0
	endif
	if (!(numpnts (mafPC_Patterns) > 0))
		return 0
	endif
	if (strlen (setname) == 0)
		setname = mafPC_SetName	// preserve old name
		NewPanel /W=(371.25,485,543,574.25)
		dowindow /c getNewSetName
		DoWindow /T getNewSetName, "Save Set As"
		SetDrawLayer UserBack
		DrawText 7,23,"Enter the new set name:"
		SetVariable setvar0,pos={39,26},size={123,19},title=" ",fSize=12
		SetVariable setvar0,value= root:maf:mafPC_SetName, proc = SaveSetFieldProc
		Button SaveButton,pos={112,57},size={50,20},proc=mafPC_OKCancelButtonProc,title="Save"
		Button CancelButton,pos={51,57},size={50,20},proc=mafPC_OKCancelButtonProc,title="Cancel"
		autopositionwindow /m=1/r=mafPC_PatternPanel
		SaveSetFieldProc ("", 0, mafPC_SetName, "")
		controlupdate setvar0
		setvariable setvar0 activate
		pauseforuser getNewSetName
		if (mafPC_userResponse == 0 || strlen (mafPC_setname) == 0) // i.e. cancel
			return 0
		endif
		open /r/T="TEXT"/Z=1 savefile as mafPC_SetName + ".set"	// test if set with that name exists
		if (v_flag == 0)	// i.e. file already exists
			close savefile
			doalert 1, "Set \"" + mafPC_SetName + "\" already exists.  Do you wish to overwrite it?"
			if (v_flag == 2)	// i.e. no
				mafPC_SetName = setname	// set back to old name
				return 0
			endif
		endif
	else
		mafPC_SetName = setname
	endif
	patlist = ""
	for (i = 0; i < numpnts (mafPC_Patterns); i += 1)
		patlist += mafPC_Patterns[i] + ";"
	endfor
	mafPC_SavePattern (patlist)
	open /T="TEXT"/p=mafPCPath savefile as mafPC_SetName + ".set"
	fbinwrite savefile, patlist	// error checking?***
	dowindow /t mafPC_PatternPanel, mafPC_SetName
	close savefile
end
function mafPC_ShowSet ()
	mafPC_Init()
	SVAR mafPC_SetName=root:maf:mafPC_SetName
	dowindow /f mafPC_PatternPanel
	if (v_flag == 0)
		execute "mafPC_PatternPanel ()"
		if (strlen (mafPC_SetName) > 0)
			dowindow /t mafPC_PatternPanel, mafPC_SetName
		else
			dowindow /t mafPC_PatternPanel, "Untitled Set"
		endif
	endif
end